/**********************************************************************
 * VPI Utility Examples -- PLI application using VPI routines
 *
 * C source for functions to access handles and properties of system 
 * task/function arguments.
 *
 * Usage: 
 *   numargs = PLIbook_numargs_vpi();
 *     Returns the number of system task/function arguments.
 *
 *   arg_handle = PLIbook_getarg_handle_vpi(arg_index_number);
 *     Returns a handke for a system task/function argument,
 *     using the index number of the argument, beginning with 1.
 *
 *   arg_val = PLIbook_getarg_intval_vpi(arg_index_number);
 *     Returns a value as an int for a system task/function argument,
 *     using the index number of the argument, beginning with 1.
 *
 *   arg_val = PLIbook_getarg_realval_vpi(arg_index_number);
 *     Returns a value as a double for a system task/function arg,
 *     using the index number of the argument, beginning with 1.
 *
 *   arg_val = PLIbook_getarg_stringval_vpi(arg_index_number);
 *     Returns a value as a string pointer for a system task/func arg,
 *     using the index number of the argument, beginning with 1.
 *
 *   string_pointer = PLIbook_vpi_get_str(property, object_handle);
 *     Returns a value of a string property, trapping a NULL return.
 *
 *   PLIbook_set_vpiworkarea(systf_handle, value);
 *     Stores a value in a work area that is unique to each instance
 *     of a system task/function.
 *
 *   value = PLIbook_get_vpiworkarea(systf_handle);
 *     Retreives the value stored in the work area for a specific
 *     instance of a system task/function.
 *
 *   string_pointer = PLIbook_fetch_type_str_vpi(type_constant);
 *     Returns the constant name for a type or fulltype property.
 * 
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *********************************************************************/

#include <stdlib.h>     /* ANSI C standard library */
#include <stdio.h>      /* ANSI C standard input/output library */
#include "vpi_user.h"   /* IEEE 1364 PLI VPI routine library  */

#define PLIbookDebug /* comment out to omit verbose debug messages */

/* Prototypes of the utility applications */
int        PLIbook_numargs_vpi();
vpiHandle  PLIbook_getarg_handle_vpi(int argNum);
int        PLIbook_getarg_intval_vpi(int argNum);
double     PLIbook_getarg_realval_vpi(int argNum);
char      *PLIbook_getarg_stringval_vpi(int argNum);
char      *PLIbook_vpi_get_str(int type, vpiHandle obj_h);
void       PLIbook_set_vpiworkarea(vpiHandle systf_h, char *data);
char      *PLIbook_get_vpiworkarea(vpiHandle systf_h);
char      *PLIbook_fetch_type_str_vpi(int type);


/**********************************************************************
 * PLIbook_numargs_vpi()
 * Counts the number of system task/function arguments.  Similar to
 * tf_nump().
 *********************************************************************/
int PLIbook_numargs_vpi()
{
  vpiHandle systf_h, arg_itr, arg_h;
  int tfnum = 0;

  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_numargs_vpi() could not obtain handle to systf call\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
    }
  #else /* if error, generate brief error message */
    if (systf_h == NULL)
      vpi_printf("ERROR: PLIbook_numargs_vpi() could not obtain handle to systf call\n");
  #endif

  arg_itr = vpi_iterate(vpiArgument, systf_h);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_numargs_vpi() could not obtain iterator to systf args\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
    }
  #else /* if error, generate brief error message */
    if (systf_h == NULL)
      vpi_printf("ERROR: PLIbook_numargs_vpi() could not obtain iterator to systf args\n");
  #endif

  while (arg_h = vpi_scan(arg_itr) ) {
    tfnum++;
  }

  return(tfnum);
}


/**********************************************************************
 * PLIbook_getarg_handle_vpi() -- Version 2
 * Obtain a handle to a system task/function argument, using the
 * argument index number.  Similar to acc_handle_tfarg().
 * This version is more efficient than Version 1 (which follows),
 * because this version allocates memory and stores the task arg
 * handles so that vpi_iterate() and vpi_scan() do not need to be
 * called each time this application is called.
 *********************************************************************/
vpiHandle PLIbook_getarg_handle_vpi(int argNum)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int i, tfnum;
  vpiHandle *arg_handle_array; /* array pointer to store arg handles */
  
  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif

  if (argNum < 1) {
    vpi_printf("ERROR: PLIbook_getarg_handle_vpi() argNum invalid\n");
    return(NULL);
  }
  
  systf_h = vpi_handle(vpiSysTfCall, NULL);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf call\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
    }
  #else /* if error, generate brief error message */
    if (systf_h == NULL) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf call\n");
      return(NULL);
    }
  #endif

  /* see if already have an array with all argument handles */
  arg_handle_array = (vpiHandle *)PLIbook_get_vpiworkarea(systf_h);
  if (arg_handle_array == NULL) {
    /* allocate an array and store all argument handles in the array */
    #ifdef PLIbookDebug /* generate verbose debug message */
      vpi_printf("PLIbook_getarg_handle_vpi() is allocating workarea storage for task args\n"); 
    #endif
    tfnum = PLIbook_numargs_vpi();
    arg_handle_array = (vpiHandle *)malloc(sizeof(vpiHandle) * tfnum);
    PLIbook_set_vpiworkarea(systf_h, (char *)arg_handle_array);

    arg_itr = vpi_iterate(vpiArgument, systf_h);
    #ifdef PLIbookDebug /* if error, generate verbose debug message */
      if (vpi_chk_error(&err)) {
        vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain iterator to systf args\n");
        vpi_printf("File %s, Line %d: %s\n",
                   err.file, err.line, err.message);
      }
    #else /* if error, generate brief error message */
      if (systf_h == NULL) {
        vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain iterator to systf args\n");
        return(NULL);
      }
    #endif
    for (i=0; i<=tfnum; i++) {
      arg_h = vpi_scan(arg_itr);
      #ifdef PLIbookDebug /* if error, generate verbose debug message */
        if (vpi_chk_error(&err)) {
          vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf arg %d\n", i);
          vpi_printf("File %s, Line %d: %s\n",
                     err.file, err.line, err.message);
        }
      #endif
      arg_handle_array[i] = arg_h;
    }
    vpi_free_object(arg_itr); /* free iterator -- didn't scan all args */
  }

  arg_h = (vpiHandle)arg_handle_array[argNum-1]; /* array starts with 0, argnums with 1 */
  return(arg_h);
}


/**********************************************************************
 * PLIbook_getarg_handle_vpi() -- Version 1
 * Obtain a handle to a system task/function argument, using the
 * argument index number.  Similar to acc_handle_tfarg().
 * The method used in this version is not as efficient as version 2
 * because this method must call vpi_iterate() and vpi_scan() for the 
 * system task args each time this application is called.
 *********************************************************************/
vpiHandle PLIbook_getarg_handle_vpi_version1(int argNum)
{
  vpiHandle systf_h, arg_itr, arg_h;
  int i;
  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif

  if (argNum < 1) {
    vpi_printf("ERROR: PLIbook_getarg_handle_vpi() argNum invalid\n");
    return(NULL);
  }

  systf_h = vpi_handle(vpiSysTfCall, NULL);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf call\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
    }
  #else /* if error, generate brief error message */
    if (systf_h == NULL) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf call\n");
      return(NULL);
    }
  #endif

  arg_itr = vpi_iterate(vpiArgument, systf_h);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain iterator to systf args\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
    }
  #else /* if error, generate brief error message */
    if (systf_h == NULL) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain iterator to systf args\n");
      return(NULL);
    }
  #endif

  for (i=1; i<=argNum; i++) {
    arg_h = vpi_scan(arg_itr);
    #ifdef PLIbookDebug /* if error, generate verbose debug message */
      if (vpi_chk_error(&err)) {
        vpi_printf("ERROR: PLIbook_getarg_handle_vpi() could not obtain handle to systf arg %d\n", i);
        vpi_printf("File %s, Line %d: %s\n",
                   err.file, err.line, err.message);
      }
    #endif
    if (arg_h == NULL) {
      vpi_printf("ERROR: PLIbook_getarg_handle_vpi() systf arg %d out-of-range\n",
                 argNum);
      return(NULL);
    }
  }
  vpi_free_object(arg_itr); /* free iterator -- didn't scan all args */

  return(arg_h);
}


/**********************************************************************
 * PLIbook_getarg_intval_vpi()
 * Return the value of a system task/function argument as an integer.
 * Similar to acc_fetch_tfarg_int()
 *********************************************************************/
int PLIbook_getarg_intval_vpi(int argNum)
{
  vpiHandle arg_h;
  s_vpi_value argVal;
  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif
 
  arg_h = PLIbook_getarg_handle_vpi(argNum);
  if (arg_h == NULL) {
    vpi_printf("ERROR: PLIbook_getarg_intval_vpi() could not obtain arg handle\n");
    return(0);
  }
  argVal.format = vpiIntVal;
  vpi_get_value(arg_h, &argVal);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_intval_vpi() could not obtain arg value\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
      return(0);
    }
  #endif

  return(argVal.value.integer);
}


/**********************************************************************
 * PLIbook_getarg_realval_vpi()
 * Return the value of a system task/function argument as a double.
 * Similar to acc_fetch_tfarg()
 *********************************************************************/
double PLIbook_getarg_realval_vpi(int argNum)
{
  vpiHandle arg_h;
  s_vpi_value argVal;
  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif
 
  arg_h = PLIbook_getarg_handle_vpi(argNum);
  if (arg_h == NULL) {
    vpi_printf("ERROR: PLIbook_getarg_realval_vpi() could not obtain arg handle\n");
    return(0);
  }
  argVal.format = vpiRealVal;
  vpi_get_value(arg_h, &argVal);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_realval_vpi() could not obtain arg value\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
      return(0.0);
    }
  #endif

  return(argVal.value.real);
}


/**********************************************************************
 * PLIbook_getarg_stringval_vpi()
 * Return the value of a system task/function argument as a string.
 * Similar to acc_fetch_tfarg_str()
 *********************************************************************/
char *PLIbook_getarg_stringval_vpi(int argNum)
{
  vpiHandle arg_h;
  s_vpi_value argVal;
  #ifdef PLIbookDebug
    s_vpi_error_info err;  /* structure for error handling */
  #endif
 
  arg_h = PLIbook_getarg_handle_vpi(argNum);
  if (arg_h == NULL) {
    vpi_printf("ERROR: PLIbook_getarg_stringval_vpi() could not obtain arg handle\n");
    return(0);
  }
  argVal.format = vpiStringVal;
  vpi_get_value(arg_h, &argVal);
  #ifdef PLIbookDebug /* if error, generate verbose debug message */
    if (vpi_chk_error(&err)) {
      vpi_printf("ERROR: PLIbook_getarg_stringval_vpi() could not obtain arg value\n");
      vpi_printf("File %s, Line %d: %s\n",
                 err.file, err.line, err.message);
      return("");
    }
  #endif

  return(argVal.value.str);
}


/**********************************************************************
 * PLIbook_vpi_get_str()
 * Return the string property of an object.  Works like vpi_get_str()
 * except that it traps a null return and converts it to a null string
 *********************************************************************/
char *PLIbook_vpi_get_str(int type, vpiHandle obj_h)
{
  char *temp;

  temp = vpi_get_str(type, obj_h);
  if (temp != NULL)
    return(temp);
  else
    return("");
}


/**********************************************************************
 * PLIbook_set_vpiworkarea()
 *   Stores a data value in the work area for a specific system
 *   task/function instance.  Similar to tf_setworkarea().
 *
 *   This routine automatically allocates a work area that is specific
 *   to an instance of a system task or system function. The work area
 *   is created using a simple LIFO stack so that a work area can be 
 *   allocated for any number of system task/function instances. Only
 *   one work area per task/function instance will be allocated.
 * 
 *   The work area stores a char* pointer in its data field. The data
 *   can be retrieved using PLIbook_set_vpiworkarea().
 *********************************************************************/

/* Work area structure definition */
typedef struct PLIbook_vpiworkarea *PLIbook_vpiworkarea_p;
typedef struct PLIbook_vpiworkarea {
  vpiHandle systf_h; /* shows which systf instance owns this space */
  char *data;        /* data to be stored in workarea */
  PLIbook_vpiworkarea_p next_workarea;
} PLIbook_vpiworkarea_s;


/* allocate a global stack pointer */
static PLIbook_vpiworkarea_p PLIbook_vpiworkarea_stack = NULL; 


void PLIbook_set_vpiworkarea(vpiHandle systf_h, char *data)
{
  PLIbook_vpiworkarea_p workarea = PLIbook_vpiworkarea_stack;

  /* locate the workarea in the stack for this task instance */
  while (workarea && (workarea->systf_h != systf_h))
    workarea = workarea->next_workarea;
    
  /* if no work area found for this task instance, create one */
  if (workarea == NULL) { 
    workarea =
          (PLIbook_vpiworkarea_p)malloc(sizeof(PLIbook_vpiworkarea_s));
    workarea->systf_h = systf_h;    /* set owner of this workarea */
    if (PLIbook_vpiworkarea_stack == NULL) {
      /* work area stack doesn't exist yet, create first location */
      workarea->next_workarea = NULL;
      PLIbook_vpiworkarea_stack = workarea;
    }
    else {
      workarea->next_workarea = PLIbook_vpiworkarea_stack;
      PLIbook_vpiworkarea_stack = workarea;
    }
  }

  /* store data in the work area */
  workarea->data = data;
  return;
}


/**********************************************************************
 * PLIbook_get_vpiworkarea()
 *   Returns the data stored in the work area for a specific system
 *   task/function instance. Similar to tf_getworkarea().  A NULL is
 *   returned if unsuccessful.
 *********************************************************************/
char *PLIbook_get_vpiworkarea(vpiHandle systf_h)
{
  PLIbook_vpiworkarea_p  workarea;

  /* locate the workarea in the stack for this task instance */
  workarea = PLIbook_vpiworkarea_stack;
  while (workarea && (workarea->systf_h != systf_h))
    workarea = workarea->next_workarea;
  if (workarea == NULL) {
    #ifdef PLIbookDebug  /* generate verbose debug message */
      vpi_printf("Warning: workarea not found for this task instance\n");
    #endif
    return(NULL);
  }
  return(workarea->data);
}


/**********************************************************************
 * PLIbook_fetch_type_str_vpi()
 * Return the string name of a VPI constant.
 * Similar to acc_fetch_type_string()
 *********************************************************************/
char *PLIbook_fetch_type_str_vpi(int type)
{
  switch (type) {
    case  vpiAlways         : return("vpiAlways");
    case  vpiAssignStmt     : return("vpiAssignStmt");
    case  vpiAssignment     : return("vpiAssignment");
    case  vpiBegin          : return("vpiBegin");
    case  vpiCase           : return("vpiCase");
    case  vpiCaseItem       : return("vpiCaseItem");
    case  vpiConstant       : return("vpiConstant");
    case  vpiContAssign     : return("vpiContAssign");
    case  vpiDeassign       : return("vpiDeassign");
    case  vpiDefParam       : return("vpiDefParam");
    case  vpiDelayControl   : return("vpiDelayControl");
    case  vpiDisable        : return("vpiDisable");
    case  vpiEventControl   : return("vpiEventControl");
    case  vpiEventStmt      : return("vpiEventStmt");
    case  vpiFor            : return("vpiFor");
    case  vpiForce          : return("vpiForce");
    case  vpiForever        : return("vpiForever");
    case  vpiFork           : return("vpiFork");
    case  vpiFuncCall       : return("vpiFuncCall");
    case  vpiFunction       : return("vpiFunction");
    case  vpiGate           : return("vpiGate");
    case  vpiIf             : return("vpiIf");
    case  vpiIfElse         : return("vpiIfElse");
    case  vpiInitial        : return("vpiInitial");
    case  vpiIntegerVar     : return("vpiIntegerVar");
    case  vpiInterModPath   : return("vpiInterModPath");
    case  vpiIterator       : return("vpiIterator");
    case  vpiIODecl         : return("vpiIODecl");
    case  vpiMemory         : return("vpiMemory");
    case  vpiMemoryWord     : return("vpiMemoryWord");
    case  vpiModPath        : return("vpiModPath");
    case  vpiModule         : return("vpiModule");
    case  vpiNamedBegin     : return("vpiNamedBegin");
    case  vpiNamedEvent     : return("vpiNamedEvent");
    case  vpiNamedFork      : return("vpiNamedFork");
    case  vpiNet            : return("vpiNet");
    case  vpiNetBit         : return("vpiNetBit");
    case  vpiNullStmt       : return("vpiNullStmt");
    case  vpiOperation      : return("vpiOperation");
    case  vpiParamAssign    : return("vpiParamAssign");
    case  vpiParameter      : return("vpiParameter");
    case  vpiPartSelect     : return("vpiPartSelect");
    case  vpiPathTerm       : return("vpiPathTerm");
    case  vpiPort           : return("vpiPort");
    case  vpiPortBit        : return("vpiPortBit");
    case  vpiPrimTerm       : return("vpiPrimTerm");
    case  vpiRealVar        : return("vpiRealVar");
    case  vpiReg            : return("vpiReg");
    case  vpiRegBit         : return("vpiRegBit");
    case  vpiRelease        : return("vpiRelease");
    case  vpiRepeat         : return("vpiRepeat");
    case  vpiRepeatControl  : return("vpiRepeatControl");
    case  vpiSchedEvent     : return("vpiSchedEvent");
    case  vpiSpecParam      : return("vpiSpecParam");
    case  vpiSwitch         : return("vpiSwitch");
    case  vpiSysFuncCall    : return("vpiSysFuncCall");
    case  vpiSysTaskCall    : return("vpiSysTaskCall");
    case  vpiTableEntry     : return("vpiTableEntry");
    case  vpiTask           : return("vpiTask");
    case  vpiTaskCall       : return("vpiTaskCall");
    case  vpiTchk           : return("vpiTchk");
    case  vpiTchkTerm       : return("vpiTchkTerm");
    case  vpiTimeVar        : return("vpiTimeVar");
    case  vpiTimeQueue      : return("vpiTimeQueue");
    case  vpiUdp            : return("vpiUdp");
    case  vpiUdpDefn        : return("vpiUdpDefn");
    case  vpiUserSystf      : return("vpiUserSystf");
    case  vpiVarSelect      : return("vpiVarSelect");
    case  vpiWait           : return("vpiWait");
    case  vpiWhile          : return("vpiWhile");
    case  vpiCondition      : return("vpiCondition");
    case  vpiDelay          : return("vpiDelay");
    case  vpiElseStmt       : return("vpiElseStmt");
    case  vpiForIncStmt     : return("vpiForIncStmt");
    case  vpiForInitStmt    : return("vpiForInitStmt");
    case  vpiHighConn       : return("vpiHighConn");
    case  vpiLhs            : return("vpiLhs");
    case  vpiIndex          : return("vpiIndex");
    case  vpiLeftRange      : return("vpiLeftRange");
    case  vpiLowConn        : return("vpiLowConn");
    case  vpiParent         : return("vpiParent");
    case  vpiRhs            : return("vpiRhs");
    case  vpiRightRange     : return("vpiRightRange");
    case  vpiScope          : return("vpiScope");
    case  vpiSysTfCall      : return("vpiSysTfCall");
    case  vpiTchkDataTerm   : return("vpiTchkDataTerm");
    case  vpiTchkNotifier   : return("vpiTchkNotifier");
    case  vpiTchkRefTerm    : return("vpiTchkRefTerm");
    case  vpiArgument       : return("vpiArgument");
    case  vpiBit            : return("vpiBit");
    case  vpiDriver         : return("vpiDriver");
    case  vpiInternalScope  : return("vpiInternalScope");
    case  vpiLoad           : return("vpiLoad");
    case  vpiModDataPathIn  : return("vpiModDataPathIn");
    case  vpiModPathIn      : return("vpiModPathIn");
    case  vpiModPathOut     : return("vpiModPathOut");
    case  vpiOperand        : return("vpiOperand");
    case  vpiPortInst       : return("vpiPortInst");
    case  vpiProcess        : return("vpiProcess");
    case  vpiVariables      : return("vpiVariables");
    case  vpiUse            : return("vpiUse");
    case  vpiExpr           : return("vpiExpr");
    case  vpiPrimitive      : return("vpiPrimitive");
    case  vpiStmt           : return("vpiStmt");
    
  /* Additional constants defined in Verilog-XL's vpi_user_cds.h */
  /***  
    case  vpiAttribute      : return("vpiAttribute");
    case  vpiCallback       : return("vpiCallback");
    case  vpiPorts          : return("vpiPorts");
    case  vpiTaskFunc       : return("vpiTaskFunc");
  ***/
    
    default                 : return("UNDEFINED TYPE");
  }
}
/*********************************************************************/
