(****************************************************************************)
(*                           the diy toolsuite                              *)
(*                                                                          *)
(* Jade Alglave, University College London, UK.                             *)
(* Luc Maranget, INRIA Paris-Rocquencourt, France.                          *)
(*                                                                          *)
(* Copyright 2014-present Institut National de Recherche en Informatique et *)
(* en Automatique and the authors. All rights reserved.                     *)
(*                                                                          *)
(* This software is governed by the CeCILL-B license under French law and   *)
(* abiding by the rules of distribution of free software. You can use,      *)
(* modify and/ or redistribute the software under the terms of the CeCILL-B *)
(* license as circulated by CEA, CNRS and INRIA at the following URL        *)
(* "http://www.cecill.info". We also give a copy in LICENSE.txt.            *)
(****************************************************************************)

{
module Make(O:LexUtils.Config) = struct
open Lexing
open LexMisc
open MIPSParser

module LU = LexUtils.Make(O)

let parse_reg name = begin match MIPSBase.parse_reg name with
  | Some r -> ARCH_REG r;
  | None -> NAME name
end

let check_name = function
  | "nop"-> NOP
  | "lui"-> LUI
  | "li" -> LI
  | "move" -> MOVE
  | "lw" -> LW
  | "ld" -> LD
  | "sw" -> SW
  | "ll" -> LL
  | "sc" -> SC
  | "sync" -> SYNC
(* ADD *)
  | "add" -> ADD
  | "addu" -> ADDU
  | "addi" -> ADDI
  | "addiu" -> ADDIU
  | "daddu" -> DADDU
  | "daddiu" -> DADDIU
(* DEXT *)
  | "dext" -> DEXT
(* Shift Left *)
  | "dsll" -> DSLL
(* SUB *)
  | "sub" -> SUB
  | "subu" -> SUBU
  | "subi" -> SUBI
  | "subiu" -> SUBIU
(* SLT *)
  | "slt" -> SLT
  | "sltu" -> SLTU
  | "slti" -> SLTI
  | "sltiu" -> SLTIU
(* AND *)
  | "and" -> AND
  | "andi" -> ANDI
(* OR *)
  | "or" -> OR
  | "ori" -> ORI
(* XOR *)
  | "xor" -> XOR
  | "xori" -> XORI
(* NOR *)
  | "nor" -> NOR
(* JUMPS *)
  | "b" -> B
  | "beq" -> BEQ
  | "bne" -> BNE
  | "blez" -> BLEZ
  | "bgtz" -> BGTZ
  | "bltz" -> BLTZ
  | "bgez" -> BGEZ
  | "jr" -> JR
  | name -> parse_reg name

}
let digit = [ '0'-'9' ]
let alpha = [ 'a'-'z' 'A'-'Z']
let name  = alpha (alpha|digit|'_' | '/' | '.' | '-')*
let num = digit+

rule token = parse
| [' ''\t''\r'] { token lexbuf }
| '\n'      { incr_lineno lexbuf; token lexbuf }
| "(*"      { LU.skip_comment lexbuf ; token lexbuf }
| '-' ? num as x { NUM (int_of_string x) }
| 'P' (num as x)
    { PROC (int_of_string x) }
| '%' (name as name) { SYMB_REG name }
| ';' { SEMI }
| ',' { COMMA }
| '|' { PIPE }
| '(' { LPAR }
| ')' { RPAR }
| ':' { COLON }
| '$' (name|num) as x
  { match MIPSBase.parse_reg x with
    | Some r -> ARCH_REG r
    | _ -> NAME x }
| name as x
  { check_name x }
| eof { EOF }
| ""  { error "MIPS lexer" lexbuf }

{
let token lexbuf =
   let tok = token lexbuf in
   if O.debug then begin
     Printf.eprintf
       "%a: Lexed '%s'\n"
       Pos.pp_pos2
       (lexeme_start_p lexbuf,lexeme_end_p lexbuf)
       (lexeme lexbuf)
   end ;
   tok
end
}
