/*
 * Guifications - The end all, be all, toaster popup plugin
 * Copyright (C) 2003-2005 Gary Kramlich
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1301, USA.
 */
#include <glib.h>
#include <string.h>

struct _GfThemeOptions {
	gchar *date_format;
	gchar *time_format;
	gchar *warning;
	gchar *ellipsis;
};

#ifdef HAVE_CONFIG_H
# include "../gf_config.h"
#endif
#include "gf_internal.h"
#include "gf_theme_ops.h"

#include <debug.h>
#include <xmlnode.h>


GfThemeOptions *
gf_theme_options_new() {
	GfThemeOptions *ops;

	ops = g_new0(GfThemeOptions, 1);

	ops->date_format = g_strdup("%x");
	ops->time_format = g_strdup("%X");
	ops->warning = g_strdup("N/A");
	ops->ellipsis = g_strdup("...");

	return ops;
}

GfThemeOptions *
gf_theme_options_new_from_xmlnode(xmlnode *node) {
	GfThemeOptions *ops;
	xmlnode *child;
	gchar *data;

	g_return_val_if_fail(node, NULL);

	ops = gf_theme_options_new();

	child = xmlnode_get_child(node, "date_format");
	if(child && (data = xmlnode_get_data(child))) {
		gf_theme_options_set_date_format(ops, data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "time_format");
	if(child && (data = xmlnode_get_data(child))) {
		gf_theme_options_set_time_format(ops, data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "warning");
	if(child && (data = xmlnode_get_data(child))) {
		gf_theme_options_set_warning(ops, data);
		g_free(data);
	}

	child = xmlnode_get_child(node, "ellipsis");
	if(child && (data = xmlnode_get_data(child))) {
		gf_theme_options_set_ellipsis(ops, data);
		g_free(data);
	}

	return ops;
}

xmlnode *
gf_theme_options_to_xmlnode(GfThemeOptions *ops) {
	xmlnode *parent, *child;

	parent = xmlnode_new("options");

	if(ops->date_format && strlen(ops->date_format)) {
		child = xmlnode_new_child(parent, "date_format");
		xmlnode_insert_data(child, ops->date_format, strlen(ops->date_format));
	}

	if(ops->time_format && strlen(ops->time_format)) {
		child = xmlnode_new_child(parent, "time_format");
		xmlnode_insert_data(child, ops->time_format, strlen(ops->time_format));
	}

	if(ops->warning && strlen(ops->warning)) {
		child = xmlnode_new_child(parent, "warning");
		xmlnode_insert_data(child, ops->warning, strlen(ops->warning));
	}

	if(ops->ellipsis && strlen(ops->ellipsis)) {
		child = xmlnode_new_child(parent, "ellipsis");
		xmlnode_insert_data(child, ops->ellipsis, strlen(ops->ellipsis));
	}

	return parent;
}

void
gf_theme_options_destroy(GfThemeOptions *ops) {
	g_return_if_fail(ops);

	if(ops->date_format)
		g_free(ops->date_format);

	if(ops->time_format)
		g_free(ops->time_format);

	if(ops->warning)
		g_free(ops->warning);

	if(ops->ellipsis)
		g_free(ops->ellipsis);

	g_free(ops);
	ops = NULL;
}

void
gf_theme_options_set_time_format(GfThemeOptions *ops, const gchar *format) {
	g_return_if_fail(ops);
	g_return_if_fail(format);

	if(ops->time_format)
		g_free(ops->time_format);

	ops->time_format = g_strdup(format);
}

const gchar *
gf_theme_options_get_time_format(GfThemeOptions *ops) {
	g_return_val_if_fail(ops, NULL);

	return ops->time_format;
}

void
gf_theme_options_set_date_format(GfThemeOptions *ops, const gchar *format) {
	g_return_if_fail(ops);
	g_return_if_fail(format);

	if(ops->date_format)
		g_free(ops->date_format);

	ops->date_format = g_strdup(format);
}

const gchar *
gf_theme_options_get_date_format(GfThemeOptions *ops) {
	g_return_val_if_fail(ops, NULL);

	return ops->date_format;
}

void
gf_theme_options_set_warning(GfThemeOptions *ops, const gchar *warning) {
	g_return_if_fail(ops);
	g_return_if_fail(warning);

	if(ops->warning)
		g_free(ops->warning);

	ops->warning = g_strdup(warning);
}

const gchar *
gf_theme_options_get_warning(GfThemeOptions *ops) {
	g_return_val_if_fail(ops, NULL);

	return ops->warning;
}

void
gf_theme_options_set_ellipsis(GfThemeOptions *ops, const gchar *ellipsis) {
	g_return_if_fail(ops);
	g_return_if_fail(ellipsis);

	if(ops->ellipsis)
		g_free(ops->ellipsis);

	ops->ellipsis = g_strdup(ellipsis);
}

const gchar *
gf_theme_options_get_ellipsis(GfThemeOptions *ops) {
	g_return_val_if_fail(ops, NULL);

	return ops->ellipsis;
}
