#ifndef H_CDW_FILE
#define H_CDW_FILE

#include <stdbool.h>

/* off_t */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>   /* ssize_t */

#include "main.h"
#include "cdw_dll.h"





/*
  Values for cdw_file_t->type.
  A regular file will be: cdw_file_t->type = CDW_FS_FILE
  A link to a file will be: cdw_file_t->type = CDW_FS_FILE | CDW_FS_LINK
*/
#define CDW_FS_NONE    0x0000  /* Initialization value. */
#define CDW_FS_DIR     0x0001
#define CDW_FS_FILE    0x0002
#define CDW_FS_LINK    0x0004
#define CDW_FS_OTHER   0x0008





typedef struct {
	/* index of letter in fullpath that is first letter of file name,
	   i.e. first letter after last slash */
	ssize_t name_start;

	/* full, non-relative path to a given file */
	char *fullpath;

	/* sometimes file fullpath can contain characters that can't be printed
	   in ncurses window, like in "Schlaflos_in_M�nchen/"; a file path
	   with such non-printable char is used to refer to file in file
	   system, but at the same time cdw must be able to print the string
	   in ncurses window; "printable_fullpath" is such a string; */
	char *printable_fullpath;

	/* Binary OR of CDW_FS_DIR and similar values. */
	unsigned int type;

	/* Size of file in bytes. Data type the same as of 'struct stat -> st_size'. */
	off_t size;

	/* Is the file a reference to parent dir, i.e. is it ".."
	   entity? */
	bool is_ref_to_parent_dir;

	/* stat() failed for this file: may be invalid file (e.g. broken link) or other reason */
	bool invalid;
} cdw_file_t;





cdw_file_t *cdw_file_new(char const * dirpath, char const * name);
cdw_rv_t    cdw_file_delete(cdw_file_t **file);

cdw_file_t *cdw_file_duplicate(cdw_file_t const * file);
bool cdw_file_equal(void const * _file1, void const * _file2);
bool cdw_file_equal_by_fullpath(void const * _fullpath, void const * _file);

void cdw_file_dealloc_files_from_list(cdw_dll_item_t * list);

void cdw_file_display_file(void *display, void *data, size_t row, size_t h_offset, bool isreverse);


char *cdw_file_shorten_fullpath(cdw_file_t const * file);




/* For unit testing. */
void cdw_file_run_tests(void);





#endif /* H_CDW_FILE */
