/******************************************************************************
 *
 * Program: NetWare statistics plugin for NetSaint
 * License: GPL
 *
 * License Information:
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: check_nwstat.c,v 1.9.2.1 2001/01/13 23:30:39 karldebisschop Exp $
 *
 *****************************************************************************/

#define PROGNAME "check_nwstat"
#define REVISION "$Revision: 1.9.2.1 $"
#define COPYRIGHT "Copyright (c) 1999-2001 Ethan Galstad"

#define SUMMARY "\
This plugin attempts to contact the MRTGEXT NLM running on a Novell server\n\
to gather the requested system information.\n"

#define OPTIONS "\
-H host [-v variable] [-w warning] [-c critical]\n\
              [-p port] [-t timeout]"

#define LONGOPTIONS "\
-H, --hostname=HOST\n\
  Name of the host to check\n\
-v, --variable=STRING\n\
  Variable to check.  Valid variables include:\n\
     LOAD1    = 1 minute average CPU load\n\
     LOAD5    = 5 minute average CPU load\n\
     LOAD15   = 15 minute average CPU load\n\
     CONNS    = number of currently licensed connections\n\
     VPF<vol> = percent free space on volume <vol>\n\
     VKF<vol> = KB of free space on volume <vol>\n\
     LTCH     = percent long term cache hits\n\
     CBUFF    = current number of cache buffers\n\
     CDBUFF   = current number of dirty cache buffers\n\
     LRUM     = LRU sitting time in minutes\n\
     DSDB     = check to see if DS Database is open\n\
     LOGINS   = check to see if logins are enabled\n\
     UPRB     = used packet receive buffers\n\
     PUPRB    = percent (of max) used packet receive buffers\n\
     SAPENTRIES = number of entries in the SAP table\n\
     SAPENTRIES<n> = number of entries in the SAP table for SAP type <n>\n\
-w, --warning=INTEGER\n\
  Threshold which will result in a warning status\n\
-c, --critical=INTEGER\n\
  Threshold which will result in a critical status\n\
-p, --port=INTEGER\n\
  Optional port number (default: %d)\n\
-t, --timeout=INTEGER\n\
  Seconds before connection attempt times out (default: %d)\n\
-h, --help\n\
  Print this help screen\n\
-V, --version\n\
  Print version information\n"

#define DESCRIPTION "\
Notes:\n\
- This plugin requres that the MRTGEXT.NLM file from James Drews' MRTG\n\
  extension for NetWare be loaded on the Novell servers you wish to check.\n\
  (available from http://www.engr.wisc.edu/~drews/mrtg/)\n\
- Values for critical thresholds should be lower than warning thresholds\n\
  when the following variables are checked: VPF, VKF, LTCH, CBUFF, and LRUM.\n"

#include "config.h"
#include "common.h"
#include "netutils.h"
#include "utils.h"

#define CHECK_NONE           0
#define CHECK_LOAD1          1
#define CHECK_LOAD5          2
#define CHECK_LOAD15         4
#define CHECK_CONNS          8
#define CHECK_VPF           16
#define CHECK_VKF           32
#define CHECK_LTCH          64
#define CHECK_CBUFF        128
#define CHECK_CDBUFF       256
#define CHECK_LRUM         512
#define CHECK_DSDB        1024 /* check to see if DS Database is open */
#define CHECK_LOGINS      2048 /* check to see if logins are enabled */
#define CHECK_PUPRB       4096 /* check % of used packet receive buffers */
#define CHECK_UPRB        8192 /* check used packet receive buffers */
#define CHECK_SAPENTRIES 16384 /* check SAP entries */

#define PORT 9999

char *server_address=NULL;
char *volume_name=NULL;
int server_port=PORT;
unsigned long warning_value=0L;
unsigned long critical_value=0L;
int check_warning_value=FALSE;
int check_critical_value=FALSE;
int vars_to_check=CHECK_NONE;
int sap_number=-1;

#define PROGNAME "check_nwstat"

int process_arguments(int, char **);
void print_usage(void);
void print_help(void);

int main(int argc, char **argv){
	int result;
	char *send_buffer=NULL;
	char recv_buffer[MAX_INPUT_BUFFER];
	char *output_message=NULL;
	char *temp_buffer=NULL;

	unsigned long free_disk_space;
	unsigned long total_disk_space;
	unsigned long percent_free_space;
	unsigned long current_connections;
	unsigned long utilization;
	unsigned long cache_hits;
	unsigned long cache_buffers;
	unsigned long lru_time;
	char uptime[MAX_INPUT_BUFFER];
	int max_packet_receive_buffers=0;
	int used_packet_receive_buffers=0;
	unsigned long percent_used_packet_receive_buffers=0L;
	int sap_entries=0;

	if(process_arguments(argc,argv)==ERROR)
		usage("Could not parse arguments\n");

	/* initialize alarm signal handling */
	signal(SIGALRM,socket_timeout_alarm_handler);

	/* set socket timeout */
	alarm(socket_timeout);

	if (vars_to_check==CHECK_LOAD1 || vars_to_check==CHECK_LOAD5 || vars_to_check==CHECK_LOAD15) {
			
		send_buffer = strscpy(send_buffer,"UTIL1\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		utilization=strtoul(recv_buffer,NULL,10);
		send_buffer = strscpy(send_buffer,"UPTIME\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		recv_buffer[strlen(recv_buffer)-1]=0;
		sprintf(uptime,"Up %s,",recv_buffer);

		if(check_critical_value==TRUE && utilization >= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && utilization >= warning_value)
			result=STATE_WARNING;

		switch(vars_to_check){
		case CHECK_LOAD1:
			temp_buffer = strscpy(temp_buffer,"1");
			break;
		case CHECK_LOAD5:
			temp_buffer = strscpy(temp_buffer,"5");
			break;
		default:
			temp_buffer = strscpy(temp_buffer,"15");
			break;
		}

		output_message = ssprintf(output_message,"Load %s - %s %s-min load average = %lu%%",(result==STATE_OK)?"ok":"problem",uptime,temp_buffer,utilization);

	} else if (vars_to_check==CHECK_CONNS) {

		send_buffer = strscpy(send_buffer,"CONNECT\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		current_connections=strtoul(recv_buffer,NULL,10);

		if(check_critical_value==TRUE && current_connections >= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && current_connections >= warning_value)
			result=STATE_WARNING;
		output_message = ssprintf(output_message,"Conns %s - %lu current connections",(result==STATE_OK)?"ok":"problem",current_connections);

	} else if (vars_to_check==CHECK_LTCH) {

		send_buffer = strscpy(send_buffer,"S1\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		cache_hits=strtoul(recv_buffer,NULL,10);

		if(check_critical_value==TRUE && cache_hits <= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && cache_hits <= warning_value)
			result=STATE_WARNING;
		output_message = ssprintf(output_message,"Long term cache hits = %lu%%",cache_hits);

	} else if (vars_to_check==CHECK_CBUFF) {

		send_buffer = strscpy(send_buffer,"S2\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		cache_buffers=strtoul(recv_buffer,NULL,10);

		if(check_critical_value==TRUE && cache_buffers <= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && cache_buffers <= warning_value)
			result=STATE_WARNING;
		output_message = ssprintf(output_message,"Total cache buffers = %lu",cache_buffers);

	} else if (vars_to_check==CHECK_CDBUFF) {

		send_buffer = strscpy(send_buffer,"S3\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		cache_buffers=strtoul(recv_buffer,NULL,10);

		if(check_critical_value==TRUE && cache_buffers >= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && cache_buffers >= warning_value)
			result=STATE_WARNING;
		output_message = ssprintf(output_message,"Dirty cache buffers = %lu",cache_buffers);

	} else if (vars_to_check==CHECK_LRUM) {

		send_buffer = strscpy(send_buffer,"S5\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		lru_time=strtoul(recv_buffer,NULL,10);

		if(check_critical_value==TRUE && lru_time <= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && lru_time <= warning_value)
			result=STATE_WARNING;
		output_message = ssprintf(output_message,"LRU sitting time = %lu minutes",lru_time);

	} else if (vars_to_check==CHECK_VKF) {

		send_buffer = ssprintf(send_buffer,"VKF%s\r\n",volume_name);
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;

		if (!strcmp(recv_buffer,"-1\n")) {
			output_message = ssprintf(output_message,"Error: Volume '%s' does not exist!",volume_name);
			result=STATE_CRITICAL;
		}	else {
			free_disk_space=strtoul(recv_buffer,NULL,10);
			if(check_critical_value==TRUE && free_disk_space <= critical_value)
				result=STATE_CRITICAL;
			else if(check_warning_value==TRUE && free_disk_space <= warning_value)
				result=STATE_WARNING;
			output_message = ssprintf(output_message,"%s%lu KB free on volume %s",(result==STATE_OK)?"":"Only ",free_disk_space,volume_name);
		}

	} else if (vars_to_check==CHECK_VPF) {

		send_buffer = ssprintf(send_buffer,"VKF%s\r\n",volume_name);
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;

		if(!strcmp(recv_buffer,"-1\n")){

			output_message = ssprintf(output_message,"Error: Volume '%s' does not exist!",volume_name);
			result=STATE_CRITICAL;

		} else {

			free_disk_space=strtoul(recv_buffer,NULL,10);

			send_buffer = ssprintf(send_buffer,"VKS%s\r\n",volume_name);
			result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
			if(result!=STATE_OK)
				return result;
			total_disk_space=strtoul(recv_buffer,NULL,10);

			percent_free_space=(unsigned long)(((double)free_disk_space/(double)total_disk_space)*100.0);

			if(check_critical_value==TRUE && percent_free_space <= critical_value)
				result=STATE_CRITICAL;
			else if(check_warning_value==TRUE && percent_free_space <= warning_value)
				result=STATE_WARNING;
			free_disk_space/=1024;
			output_message = ssprintf(output_message,"%lu MB (%lu%%) free on volume %s",free_disk_space,percent_free_space,volume_name);
		}

	/* check to see if DS Database is open or closed */
	} else if(vars_to_check==CHECK_DSDB) {

		send_buffer = strscpy(send_buffer,"S11\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		if(atoi(recv_buffer)==1)
			result=STATE_OK;
		else
			result=STATE_WARNING;
 
		send_buffer = strscpy(send_buffer,"S13\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		temp_buffer=strtok(recv_buffer,"\r\n");
 
		output_message = ssprintf(output_message,"Directory Services Database is %s (DS version %s)",(result==STATE_OK)?"open":"closed",temp_buffer);

	/* check to see if logins are enabled */
	} else if (vars_to_check==CHECK_LOGINS) {

		send_buffer = strscpy(send_buffer,"S12\r\n");
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
		if(atoi(recv_buffer)==1)
			result=STATE_OK;
		else
			result=STATE_WARNING;
 
		output_message = ssprintf(output_message,"Logins are %s",(result==STATE_OK)?"enabled":"disabled");

	/* check packet receive buffers */
	} else if (vars_to_check==CHECK_UPRB || vars_to_check==CHECK_PUPRB) {
 
		send_buffer = ssprintf(send_buffer,"S15\r\n",volume_name);
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;

		used_packet_receive_buffers=atoi(recv_buffer);

		send_buffer = ssprintf(send_buffer,"S16\r\n",volume_name);
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;

		max_packet_receive_buffers=atoi(recv_buffer);
 
		percent_used_packet_receive_buffers=(unsigned long)(((double)used_packet_receive_buffers/(double)max_packet_receive_buffers)*100.0);

		if(vars_to_check==CHECK_UPRB){
			if(check_critical_value==TRUE && used_packet_receive_buffers >= critical_value)
				result=STATE_CRITICAL;
			else if(check_warning_value==TRUE && used_packet_receive_buffers >= warning_value)
				result=STATE_WARNING;
		} else {
			if(check_critical_value==TRUE && percent_used_packet_receive_buffers >= critical_value)
				result=STATE_CRITICAL;
			else if(check_warning_value==TRUE && percent_used_packet_receive_buffers >= warning_value)
				result=STATE_WARNING;
		}
 
		output_message = ssprintf(output_message,"%d of %d (%lu%%) packet receive buffers used",used_packet_receive_buffers,max_packet_receive_buffers,percent_used_packet_receive_buffers);

	/* check SAP table entries */
	} else if (vars_to_check==CHECK_SAPENTRIES) {

		if(sap_number==-1)
			send_buffer = ssprintf(send_buffer,"S9\r\n");
		else
			send_buffer = ssprintf(send_buffer,"S9.%d\r\n",sap_number);
		result=process_tcp_request(server_address,server_port,send_buffer,recv_buffer,sizeof(recv_buffer));
		if(result!=STATE_OK)
			return result;
 
		sap_entries=atoi(recv_buffer);
 
		if(check_critical_value==TRUE && sap_entries >= critical_value)
			result=STATE_CRITICAL;
		else if(check_warning_value==TRUE && sap_entries >= warning_value)
			result=STATE_WARNING;

		if(sap_number==-1)
			output_message = ssprintf(output_message,"%d entries in SAP table",sap_entries);
		else
			output_message = ssprintf(output_message,"%d entries in SAP table for SAP type %d",sap_entries,sap_number);

	} else {

		output_message = strscpy(output_message,"Nothing to check!\n");
		result=STATE_UNKNOWN;

	}

	/* reset timeout */
	alarm(0);

	printf("%s\n",output_message);

	return result;
}


/* process command-line arguments */
int process_arguments(int argc, char **argv){
	int c;

#ifdef HAVE_GETOPT_H
	int option_index = 0;
	static struct option long_options[] =
	{ 
		{"port",     required_argument,0,'p'},
		{"timeout",  required_argument,0,'t'},
		{"critical", required_argument,0,'c'},
		{"warning",  required_argument,0,'w'},
		{"variable", required_argument,0,'v'},
		{"hostname", required_argument,0,'H'},
		{"version",  no_argument,      0,'V'},
		{"help",     no_argument,      0,'h'},
		{0,0,0,0}
	};
#endif

	/* no options were supplied */
	if(argc<2) return ERROR;

	/* backwards compatibility */
	if (! is_option(argv[1])) {
		server_address=argv[1];
		argv[1]=argv[0];
		argv=&argv[1];
		argc--;
	}

  for (c=1;c<argc;c++) {
    if(strcmp("-to",argv[c])==0)
      strcpy(argv[c],"-t");
    else if (strcmp("-wv",argv[c])==0)
      strcpy(argv[c],"-w");
    else if (strcmp("-cv",argv[c])==0)
      strcpy(argv[c],"-c");
	}

	while (1){
#ifdef HAVE_GETOPT_H
		c = getopt_long(argc,argv,"+hVH:t:c:w:p:v:",long_options,&option_index);
#else
		c = getopt(argc,argv,"+hVH:t:c:w:p:v:");
#endif

		if (c==-1||c==EOF||c==1)
			break;

		switch (c)
			{
			case '?': /* print short usage statement if args not parsable */
				printf("%s: Unknown argument: %s\n\n",my_basename(argv[0]),optarg);
				print_usage();
				exit(STATE_UNKNOWN);
			case 'h': /* help */
				print_help();
				exit(STATE_OK);
			case 'V': /* version */
				print_revision(my_basename(argv[0]),"$Revision: 1.9.2.1 $");
				exit(STATE_OK);
			case 'H': /* hostname */
				server_address=optarg;
				break;
			case 'p': /* port */
				if (is_intnonneg(optarg))
					server_port=atoi(optarg);
				else
					terminate(STATE_UNKNOWN,"Server port an integer (seconds)\nType '%s -h' for additional help\n",PROGNAME);
				break;
			case 'v':
				if(strlen(optarg)<4)
					return ERROR;
				if(!strcmp(optarg,"LOAD1"))
					vars_to_check=CHECK_LOAD1;
				else if(!strcmp(optarg,"LOAD5"))
					vars_to_check=CHECK_LOAD5;
				else if(!strcmp(optarg,"LOAD15"))
					vars_to_check=CHECK_LOAD15;
				else if(!strcmp(optarg,"CONNS"))
					vars_to_check=CHECK_CONNS;
				else if(!strcmp(optarg,"LTCH"))
					vars_to_check=CHECK_LTCH;
				else if(!strcmp(optarg,"CBUFF"))
					vars_to_check=CHECK_CBUFF;
				else if(!strcmp(optarg,"CDBUFF"))
					vars_to_check=CHECK_CDBUFF;
				else if(!strcmp(optarg,"LRUM"))
					vars_to_check=CHECK_LRUM;
				else if(strncmp(optarg,"VPF",3)==0){
					vars_to_check=CHECK_VPF;
					volume_name = strscpy(volume_name,optarg+3);
				}
				else if(strncmp(optarg,"VKF",3)==0){
					vars_to_check=CHECK_VKF;
					volume_name = strscpy(volume_name,optarg+3);
				}
				else if(!strcmp(optarg,"DSDB"))
					vars_to_check=CHECK_DSDB;
				else if(!strcmp(optarg,"LOGINS"))
					vars_to_check=CHECK_LOGINS;
				else if(!strcmp(optarg,"UPRB"))
					vars_to_check=CHECK_UPRB;
				else if(!strcmp(optarg,"PUPRB"))
					vars_to_check=CHECK_PUPRB;
				else if(!strncmp(optarg,"SAPENTRIES",10)){
					vars_to_check=CHECK_SAPENTRIES;
					if(strlen(optarg)>10)
						sap_number=atoi(optarg+10);
					else
						sap_number=-1;
				}
				else
					return ERROR;
				break;
			case 'w': /* warning threshold */
				warning_value=strtoul(optarg,NULL,10);
				check_warning_value=TRUE;
				break;
			case 'c': /* critical threshold */
				critical_value=strtoul(optarg,NULL,10);
				check_critical_value=TRUE;
				break;
			case 't': /* timeout */
				socket_timeout=atoi(optarg);
				if(socket_timeout<=0)
					return ERROR;
			}

	}

	return OK;
}


void print_usage(void)
{
	printf
		("Usage:\n"
		 " %s %s\n"
#ifdef HAVE_GETOPT_H
		 " %s (-h | --help) for detailed help\n"
		 " %s (-V | --version) for version information\n",
#else
		 " %s -h for detailed help\n"
		 " %s -V for version information\n",
#endif
		 PROGNAME, OPTIONS, PROGNAME, PROGNAME);
}

void print_help(void)
{
	print_revision (PROGNAME, REVISION);
	printf ("%s\n\n%s\n", COPYRIGHT, SUMMARY);
	print_usage();
	printf
		("\nOptions:\n" LONGOPTIONS "\n" DESCRIPTION "\n",
		 PORT, DEFAULT_SOCKET_TIMEOUT);
	support ();
}
