/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "TreeModelMatchCollector.h"

#include <QAbstractItemModel>
#include <QDebug>
#include <QRegExp>

TreeModelMatchCollector::TreeModelMatchCollector() :
    model_(0),
    column_(0),
    flags_(Qt::MatchRecursive),
    startItemPos_(-1),
    startItemFound_(false),
    startIndexPassed_(false) {}

void TreeModelMatchCollector::setModel(QAbstractItemModel* model)
{
    model_ = model;
}

bool TreeModelMatchCollector::isSameAsStartIndex(const QModelIndex& idx) const
{
    return idx.row() == startIndex_.row() && idx.parent() == startIndex_.parent();
}

void TreeModelMatchCollector::clear()
{
    startIndex_       = QModelIndex();
    startItemPos_     = -1;
    startItemFound_   = false;
    startIndexPassed_ = false;
    value_.clear();
    column_ = 0;
    flags_  = Qt::MatchRecursive;
    items_.clear();
}

void TreeModelMatchCollector::match(QString value, int column, Qt::MatchFlags flags,
                                    const QModelIndex& startIndex)
{
    if (!model_)
        clear();

    if (value_ == value && column_ == column && flags_ == flags)
        return;

    startIndex_       = startIndex;
    startItemPos_     = -1;
    startItemFound_   = false;
    startIndexPassed_ = false;

    value_  = value;
    column_ = column_;
    flags_  = flags;
    items_.clear();

    for (int i = 0; i < model_->rowCount(); i++)
        findMatch(model_->index(i, column_));

    if (startItemPos_ == -1 && items_.count() > 0)
        startItemPos_ = 0;
}

void TreeModelMatchCollector::findMatch(const QModelIndex& idx)
{
    QString s = idx.data(Qt::DisplayRole).toString();

    bool found = false;
    if (flags_.testFlag(Qt::MatchWildcard)) {
        QRegExp rx(value_);
        rx.setPatternSyntax(QRegExp::Wildcard);
        if (flags_.testFlag(Qt::MatchCaseSensitive))
            rx.setCaseSensitivity(Qt::CaseSensitive);

        if (s.contains(rx))
            found = true;
    }
    else if (flags_.testFlag(Qt::MatchRegExp) ){
        QRegExp rx(value_);
        if (flags_.testFlag(Qt::MatchCaseSensitive))
            rx.setCaseSensitivity(Qt::CaseSensitive);

        if (s.contains(rx))
            found = true;
    }
    else {
        if (s.contains(value_,
                       (flags_.testFlag(Qt::MatchCaseSensitive)) ? Qt::CaseSensitive : Qt::CaseInsensitive)) {            
            found = true;
        }
    }

    if (found) {
        items_ << idx;
    }

    if (idx == startIndex_) {
        startItemPos_     = items_.count() - 1;
        startIndexPassed_ = true;
        if (found)
            startItemFound_ = true;
    }
    else if (startIndexPassed_ && !startItemFound_ && found) {
        startItemPos_   = items_.count() - 1;
        startItemFound_ = true;
    }

    int chNum = idx.model()->rowCount(idx);
    for (int i = 0; i < chNum; i++) {
        findMatch(idx.child(i, column_));
    }
}

void TreeModelMatchCollector::allValues(QSet<QString>& vals) const
{
    for (int i = 0; i < model_->rowCount(); i++)
        allValues(model_->index(i, column_), vals);
}

void TreeModelMatchCollector::allValues(const QModelIndex& idx, QSet<QString>& vals) const
{
    vals << idx.data(Qt::DisplayRole).toString();

    int chNum = idx.model()->rowCount(idx);
    for (int i = 0; i < chNum; i++)
        allValues(idx.child(i, column_), vals);
}
