/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * faketape
 */

#include "faketape-lib.h"

#include <err.h>	// errx
#include <string.h>	// strerror
#include <fstream>	// ofstream
#include <charconv>	// from_chars
#include <fcntl.h>	// O_ACCMODE
#include <sys/mtio.h>
extern "C" {
#include <rmtflags.h>
#define USE_QFA
#include "../rmt/rmt.h"
}

FakeTape	tape;

std::vector<uint8_t> databuffer;

template<typename T> static bool convertstring(const std::string& s, T& v);

static void	getstring (std::string& s);

static bool respond(off_t val, FakeTape::Response r = FakeTape::Response::OK) {
	static char resp[BUFSIZ];
	const char* errstr = strerror(val);
	char respchar;
	switch(r) {
		case FakeTape::Response::OK:
			respchar = 'A';
			errstr = "";
			break;
		case FakeTape::Response::ERROR:
			respchar = 'E';
			break;
		case FakeTape::Response::FATAL:
			respchar = 'F';
			break;
		default:
			errx(1, "Unhandled Response");
	}
	DEBUG("faketape response: " << respchar << " " << val << " " << errstr);
	(void)snprintf(resp, sizeof(resp), "%c%jd\n%s%c", respchar, (intmax_t)val, errstr, *errstr?'\n':0);
	if (write(1, resp, strlen(resp)) != (ssize_t)strlen(resp)) {
		warn("faketape: write error sending response\n");
		return false;
	}
	return true;
}

static bool ioerror(FakeTape::Response r)
{
	return respond(errno, r);
}

std::unique_ptr<FakeTapeTimer> process_time;
std::unique_ptr<FakeTapeTimer> open_time;
std::unique_ptr<FakeTapeTimer> close_time;
std::unique_ptr<FakeTapeTimer> write_time;
std::unique_ptr<FakeTapeTimer> read_time;

bool process_command()
{
	CalcTime ct(*process_time);
	using Response = FakeTape::Response;
	errno = 0;
	char c;
	if (read(0, &c, 1) != 1)
		return false;
	switch (c) {

	case 'O':
	{
		CalcTime ct(*open_time);
		std::string device;
		std::string filemode;
		getstring(device);
		getstring(filemode);
		DEBUG("faketape: O " << device << " " << filemode << std::endl);
		tape.close();
		int oflags = rmtflags_toint(filemode.c_str());
		if(Response r = tape.open(device, oflags & O_ACCMODE); r != Response::OK)
			return ioerror(r);
		else
			return respond(0);
	}

	case 'C':
	{
		CalcTime ct(*close_time);
		std::string device;
		getstring(device);
		DEBUG("faketape: C " << device << " (" << tape.getPos() << " blocks)");
		tape.close();
		return respond(0);
	}

	case 'L':
	{
		std::string count;
		std::string pos;
		getstring(count);
		getstring(pos);
		DEBUG("faketape: L " << count << " " << pos);

		off_t offset;
		int cmd;
		if (!convertstring(count, offset) || !convertstring(pos, cmd))
			return ioerror(Response::ERROR);

		switch (cmd) {
			case SEEK_SET:
				tape.blockSeek(offset);
				return respond(tape.getPos());
			case SEEK_CUR:
				offset += tape.getPos();
				tape.blockSeek(offset);
				return respond(tape.getPos());
			case LSEEK_GET_TAPEPOS:
				return respond(tape.getPos());
			case LSEEK_IS_MAGTAPE:
				return respond(1);
			case LSEEK_GO2_TAPEPOS:
				tape.blockSeek(offset);
				return respond(tape.getPos());
			case SEEK_END:
				/* Too hard to do */
			default:
				errno = EINVAL;
				return ioerror(Response::ERROR);
		}
	}

	case 'W':
	{
		CalcTime ct(*write_time);
		std::string count;
		getstring(count);
		DEBUG("faketape: W " << count << " (block = " << tape.getPos() << ")");

		size_t size;
		if (!convertstring(count, size)) {
			return ioerror(Response::ERROR);
		}

		if (databuffer.size() < size)
			databuffer.resize(size);

		ssize_t cc;
		for (size_t i = 0; i < size; i += cc) {
			cc = read(0, databuffer.data()+i, size - i);
			if (cc <= 0) {
				DEBUG("faketape: premature eof i=" << i);
				return false;
			}
		}

		if(Response r = tape.writeDataAsync(databuffer.data(), size); r != Response::OK)
			return ioerror(r);

		return respond(size);
	}

	case 'R':
	{
		CalcTime ct(*read_time);
		std::string count;
		getstring(count);
		DEBUG("faketape: R " << count << " (block " << tape.getPos() << ")");

		size_t size;
		if (!convertstring(count, size))
			return ioerror(Response::ERROR);

		if (databuffer.size() < size)
			databuffer.resize(size);

		size_t readsize;
		if(Response r = tape.readData(databuffer.data(), size, &readsize); r != Response::OK)
			return respond(EIO, r);


		if(!respond(readsize))
			return false;
		ssize_t cc;
		for (size_t i = 0; i < readsize; i += cc) {
			cc = write(1, databuffer.data(), readsize);
			if (cc <= 0) {
				DEBUG("faketape: failed to write response");
				return false;
			}
		}
		return true;
	}

	case 'I':
	{
		std::string sop;
		std::string scount;
		getstring(sop);
		getstring(scount);
		DEBUG("faketape: I " << sop << " " << scount);

		size_t count;
		int op;
		if (!convertstring(scount, count) || ! convertstring(sop, op))
			return ioerror(Response::ERROR);

		off_t rval = -1;
		switch(op) {
			case MTRESET:		/* 0	+reset drive in case of problems.  */
			case MTFSF:		/* 1	Forward space over FileMark,
							position at first record of next file.  */
			case MTBSF:		/* 2	Backward space FileMark (position before FM).  */
			case MTFSR:		/* 3	Forward space record.  */
			case MTBSR:		/* 4	Backward space record.  */
			case MTWEOF:		/* 5	Write an end-of-file record (mark).  */
			case MTREW:		/* 6	Rewind.  */
			case MTOFFL:		/* 7	Rewind and put the drive offline (eject?).  */
			case MTNOP:		/* 8	No op, set status only (read with MTIOCGET).  */
			case MTRETEN:		/* 9	Retension tape.  */
			case MTBSFM:		/* 10	+backward space FileMark, position at FM.  */
			case MTFSFM:		/* 11	+forward space FileMark, position at FM.  */
			case MTEOM:		/* 12	Goto end of recorded media (for appending files).
							MTEOM positions after the last FM, ready for
							appending another file.  */
			case MTERASE:		/* 13	Erase tape -- be careful!  */

			case MTRAS1:		/* 14	Run self test 1 (nondestructive).  */
			case MTRAS2:		/* 15	Run self test	2 (destructive).  */
			case MTRAS3:		/* 16	Reserved for self test 3.  */

			case MTSETBLK:		/* 20	Set block length (SCSI).  */
			case MTSETDENSITY:	/* 21	Set tape density (SCSI).  */
			case MTSEEK:		/* 22	Seek to block (Tandberg, etc.).  */
			case MTTELL:		/* 23	Tell block (Tandberg, etc.).  */
			case MTSETDRVBUFFER:	/* 24	Set the drive buffering according to SCSI-2.
							Ordinary buffered operation with code	1.  */
			case MTFSS:		/* 25	Space forward over setmarks.  */
			case MTBSS:		/* 26	Space backward over setmarks.  */
			case MTWSM:		/* 27	Write setmarks.  */

			case MTLOCK:		/* 28	Lock the drive door.  */
			case MTUNLOCK:		/* 29	Unlock the drive door.  */
			case MTLOAD:		/* 30	Execute the SCSI load command.  */
			case MTUNLOAD:		/* 31	Execute the SCSI unload command.  */
			case MTCOMPRESSION:	/* 32	Control compression with SCSI mode page 15.  */
			case MTSETPART:		/* 33	Change the active tape partition.  */
			case MTMKPART:		/* 34	Format the tape with one or two partitions.  */
			default:
				errno = EINVAL;
				return respond(rval);
		}
	}

	case 'S': /* status */
	{
		DEBUG("faketape: S\n");
		struct mtget mtget;
		mtget.mt_type = 0;
		mtget.mt_resid = 0;
		mtget.mt_dsreg = 0;
		mtget.mt_gstat = 0;
		mtget.mt_erreg = 0;
		mtget.mt_fileno = 0;
		mtget.mt_blkno = 0;

		if (!respond(sizeof (mtget)))
			return false;
		if (write(1, &mtget, sizeof (mtget)) != sizeof(mtget)) {
			DEBUG("Status write failed");
			return false;
			}
		return true;
	}

	case 'V':	/* version */
	{
		std::string op;
		getstring(op);
		DEBUG("faketape: V " << op);
		return respond(2);
	}

	default:
		DEBUG("faketape: garbage command " << c);
		return false;
	}
}

void usage(const char* prog) {
	std::cerr << "Usage " << prog << " [ -b <max block size> ] [ <debug log file> ]" << std::endl;
	exit(1);
}

int
main(int argc, char *argv[])
{
	const char* prog = argv[0];

	process_time = std::make_unique<FakeTapeTimer>("process_time");
	open_time =    std::make_unique<FakeTapeTimer>("open_time");
	close_time =   std::make_unique<FakeTapeTimer>("close_time");
	write_time =   std::make_unique<FakeTapeTimer>("write_time");
	read_time =    std::make_unique<FakeTapeTimer>("read_time");

	int ch;
	while (( ch = getopt(argc, argv, "b:")) != -1)
		switch(ch) {

			case 'b': {
				ssize_t max_block_size = atoi(optarg);
				if (max_block_size < 1024 || max_block_size > 16*1024*1024)
					errx(1, "Invalid blocksize, must be between 1024 and %d", 16*1024*1024);
				tape.setMaxBlockSize(max_block_size);
				break;
			}
			default:
				std::cerr << "Unexpected " << ch << std::endl;
				usage(prog);
		}
	argc -= optind;
	argv += optind;

	if (argc > 1)
		usage(prog);

	if (argc == 1) {
		tape.debug = std::make_unique<std::ofstream>();
		tape.debug->rdbuf()->pubsetbuf(0, 0);
		static_cast<std::ofstream*>(tape.debug.get())->open(argv[0]);
		if (!tape.debug)
			err(1, "Can't open debug log %s",  argv[0]);
	}

	while(process_command())
		/* do nothing */ ;

	process_time.reset();
	open_time.reset();
	close_time.reset();
	write_time.reset();
	read_time.reset();

	if (tape.debug)
		static_cast<std::ofstream*>(tape.debug.get())->close();
}

static void getstring(std::string& s)
{
	while(true) {
		char cp;
		if (read(0, &cp, 1) != 1)
			err(1, "EOF while reading parameter: aborting");
		if (cp == '\n')
			break;
		s += cp;
	}
}


template<typename T>
static bool convertstring(const std::string& s, T& v) {
	auto [ ptr, ec ] = std::from_chars(s.data(), s.data()+s.size(), v);
	if (ec != std::errc{}) {
		errno = EINVAL;
		return false;
	}
	return true;
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
