/*************************************************************
 *
 * Program: check_pgsql.c
 * License: GPL
 *
 * Written by Karl DeBisschop
 *  (based on plugins by Ethan Galstad and PostgresQL example code)
 * Last Modified: $Date: 2001/01/25 16:48:00 $
 *
 * Command line: check_pgsql <warn_threshold> <host> <port> <database>
 *
 * Description:
 * This plugin will attempt to connect to the specified database
 * on the host.  Successul connects return STATE_OK, refusals
 * and timeouts return STATE_CRITICAL, other errors return
 * STATE_UNKNOWN.
 *
 * Modifications:
 * 08-18-1999 Ethan Galstad (netsaint@linuxbox.com)
 *	      Code now uses common include file
 *
 **************************************************************/

#define DEFAULT_DB "template1"
#define DEFAULT_PORT "5432"
#define DEFAULT_WARN 2
#define DEFAULT_CRIT 8
#define DEFAULT_TIMEOUT 30
#define DB_NAMELEN 32
#define LOGNAMELEN 32

#include "config.h"
#include "common.h"
#include "utils.h"
#include <netdb.h>
#include <libpq-fe.h>

int process_arguments(int, char **);
int call_getopt(int, char **);
int validate_arguments(void);
void print_usage(char *);
void print_help(char *);

int is_pg_dbname(char *);
int is_pg_logname(char *);

char *pghost=NULL;       /* host name of the backend server */
char *pgport=NULL;       /* port of the backend server */
char default_port[4]=DEFAULT_PORT;
char *pgoptions=NULL;
char *pgtty=NULL;
char dbName[DB_NAMELEN]=DEFAULT_DB;
char *pguser=NULL;
char *pgpasswd=NULL;
int twarn=-1;
int tcrit=-1;

PGconn     *conn;
/*PGresult   *res;*/

int main(int argc, char **argv) {
  int elapsed_time;

  /* begin, by setting the parameters for a backend connection if the
   * parameters are null, then the system will try to use reasonable
   * defaults by looking up environment variables or, failing that,
   * using hardwired constants */

  pgoptions = NULL;    /* special options to start up the backend server */
  pgtty = NULL;        /* debugging tty for the backend server */

  if(process_arguments(argc,argv)==ERROR){
    print_usage(my_basename(argv[0]));
    exit(STATE_UNKNOWN);
  }

  /* Set signal handling and alarm */
  if(signal(SIGALRM,timeout_alarm_handler)==SIG_ERR){
    printf("Cannot catch SIGALRM");
    return STATE_UNKNOWN;
  }
  alarm(timeout_interval);

  /* make a connection to the database */
  time(&start_time);
  conn = PQsetdbLogin(pghost, pgport, pgoptions, pgtty, dbName, pguser, pgpasswd);
  time(&end_time);
  elapsed_time = (int)(end_time - start_time);

  /* check to see that the backend connection was successfully made */
  if (PQstatus(conn) == CONNECTION_BAD) {
    printf("PGSQL: CRITICAL - no connection to '%s' (%s).\n", dbName,PQerrorMessage(conn));
    PQfinish(conn);
    return STATE_CRITICAL;
  } else if (elapsed_time > tcrit) {
    PQfinish(conn);
    printf("PGSQL: CRITICAL - database %s (%d sec.)\n",dbName,elapsed_time);
    return STATE_CRITICAL;
  } else if (elapsed_time > twarn) {
    PQfinish(conn);
    printf("PGSQL: WARNING - database %s (%d sec.)\n",dbName,elapsed_time);
    return STATE_WARNING;
  } else {
    PQfinish(conn);
    printf("PGSQL: ok - database %s (%d sec.)\n",dbName,elapsed_time);
    return STATE_OK;
  }
}

void print_help(char *cmd) {
  print_revision(cmd,"$Revision 0 $");
  printf("Copyright (c) 1999 Karl DeBisschop\n\n");
  printf("Usage: %s (options...)\n",cmd);
  printf("  -w <twarn>  = connection time for warning status (%d is default)\n",DEFAULT_WARN);
  printf("  -c <tcrit>  = connection time for critical status (%d is default)\n",DEFAULT_CRIT);
  printf("  -t <timeout>= timeout limit (%d is default)\n",DEFAULT_TIMEOUT);
  printf("  -H <host>   = name or ip address of machine running backend\n");
  printf("  -p <port>   = port running backend (%s is default)\n",DEFAULT_PORT);
  printf("  -d <db>     = database to check (%s is default)\n",DEFAULT_DB);
  printf("  -l <user>   = login name of user\n");
  printf("  -a <passwd> = authorization password (BIG SECURITY ISSUE)\n\n");
  printf("All parameters are optional.  The default is equivalent to:\n\n");
  printf("   %s -w 2 -c 8 -t 120 -H 127.0.0.1 -p 5432 -d template1\n\n",cmd);
  printf("and assumes that the user that runs the plugin (typically \n");
  printf("netsaint) can connect to the database without a password.\n");
  printf("This requires that the backend for remote machines uses TCP/IP\n");
  printf("(start postmaster with the -i noption)\n\n\n");
}

void print_usage(char *cmd) {
  printf("Usage: %s (options...)\n",cmd);
  printf("  -w <twarn>  = connection time for warning status (%d is default)\n",DEFAULT_WARN);
  printf("  -c <tcrit>  = connection time for critical status (%d is default)\n",DEFAULT_CRIT);
  printf("  -t <timeout>= timeout limit (%d is default)\n",DEFAULT_TIMEOUT);
  printf("  -H <host>   = name or ip address of machine running backend\n");
  printf("  -p <port>   = port running backend (%s is default)\n",DEFAULT_PORT);
  printf("  -d <db>     = database to check (%s is default)\n",DEFAULT_DB);
  printf("  -l <user>   = login name of user\n");
  printf("  -a <passwd> = authorization password (BIG SECURITY ISSUE)\n\n");
}

/* process command-line arguments */
int process_arguments(int argc, char **argv){
  int c;

  if(argc<2)
    return ERROR;

  for(c=1;c<argc;c++)
    if(strcmp("-to",argv[c])==0)
      strcpy(argv[c],"-t");

  c=0;
  while(c+=(call_getopt(argc-c,&argv[c]))){
    if(argc <= c)
      break;
    if(twarn==-1){
      if(is_intnonneg(argv[c])){
	twarn=atoi(argv[c]);
      }else{
	printf("Invalid warning threshold: %s\n",argv[c]);
	return ERROR;
      }
    } else if(tcrit==-1) {
      if(is_intnonneg(argv[c])){
	tcrit=atoi(argv[c]);
      }else{
	printf("Invalid critical threshold: %s\n",argv[c]);
	return ERROR;
      }
    } else if(pghost==NULL) {
      if(is_host(argv[c])){
	pghost=argv[c];
      }else{
	printf("Invalid hostname: %s\n",argv[c]);
	return ERROR;
      }
    } else if(pguser==NULL) {
      if(is_pg_logname(argv[c])){
	pguser=argv[c];
      }else{
	printf("Invalid user name: %s\n",argv[c]);
	return ERROR;
      }
    } else if(pgpasswd==NULL) {
      pgpasswd=argv[c];
    } else if(pgport==NULL) {
      if(is_integer(argv[c])){
	pgport=argv[c];
      }else{
	printf("Invalid service port: %s\n",argv[c]);
	return ERROR;
      }
    }
  }

  if (twarn==-1) twarn=DEFAULT_WARN;
  if (tcrit==-1) tcrit=DEFAULT_CRIT;
  if (pgport==NULL) pgport=default_port;
  return c;
}

int call_getopt(int argc, char **argv){
  int c,i=1;

#ifdef HAVE_GETOPT_H
  int option_index=0;
  static struct option long_options[] =
  { 
    {"help",   	     no_argument,0,'h'},
    {"version",	     no_argument,0,'V'},
    {"timeout",	     required_argument,0,'t'},
    {"critical",     required_argument,0,'c'},
    {"warning",      required_argument,0,'w'},
    {"hostname",     required_argument,0,'H'},
    {"logname",      required_argument,0,'l'},
    {"authorization",required_argument,0,'a'},
    {"password",     required_argument,0,'a'},
    {"port",         required_argument,0,'p'},
    {"database",     required_argument,0,'d'},
    {0,0,0,0}
  };
#endif

  while (1) {
#ifdef HAVE_GETOPT_H
    c = getopt_long(argc,argv,"+?hVt:c:w:H:p:d:l:a:",long_options,&option_index);
#else
    c = getopt(argc,argv,"+?hVt:c:w:H:p:d:l:a:");
#endif
    if(c==EOF)
      break;

    i++;
    switch (c)
      {
      case 't':
      case 'c':
      case 'w':
      case 'H':
      case 'p':
      case 'd':
      case 'l':
      case 'a':
	i++;
      }

    switch (c)
      {
      case '?': /* help */
	printf("%s: Unknown argument: %s\n\n",my_basename(argv[0]),optarg);
	print_usage(my_basename(argv[0]));
	exit(STATE_UNKNOWN);
      case 'h': /* help */
	print_help(my_basename(argv[0]));
	exit(STATE_OK);
      case 'V': /* version */
	print_revision(my_basename(argv[0]),"$Revision: 1.14.2.2 $");
	exit(STATE_OK);
      case 't': /* timeout period */
	if(!is_integer(optarg)){
	  printf("%s: Timeout Interval must be an integer!\n\n",my_basename(argv[0]));
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	timeout_interval = atoi(optarg);
	break;
      case 'c': /* critical time threshold */
	if(!is_integer(optarg)){
	  printf("Invalid critical threshold: %s\n",optarg);
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	tcrit = atoi(optarg);
	break;
      case 'w': /* warning time threshold */
	if(!is_integer(optarg)){
	  printf("Invalid critical threshold: %s\n",optarg);
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	twarn = atoi(optarg);
	break;
      case 'H': /* host */
	if(!is_host(optarg)){
	  printf("%s: You gave an invalid host name!\n\n",my_basename(argv[0]));
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	pghost=optarg;
	break;
      case 'p': /* port */
	if(!is_integer(optarg)){
	  printf("%s: Connection Port (%s) must be an integer!\n\n",my_basename(argv[0]),optarg);
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	pgport=optarg;
	break;
      case 'd': /* database name */
	if(!is_pg_dbname(optarg)){
	  printf("%s: Database Name (%s) is not valid!\n\n",my_basename(argv[0]),optarg);
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	strncpy(dbName,optarg,DB_NAMELEN-1);
	dbName[DB_NAMELEN-1]=0;
	break;
      case 'l': /* login name */
	if(!is_pg_logname(optarg)){
	  printf("%s: User Name (%s) is not valid!\n\n",my_basename(argv[0]),optarg);
	  print_usage(my_basename(argv[0]));
	  exit(STATE_UNKNOWN);
	}
	pguser=optarg;
	break;
      case 'a': /* authentication password */
	pgpasswd=optarg;
	break;
      }
  }
  return i;
}

int validate_arguments(){
  return OK;
}

int is_pg_dbname(char *dbname){
	char txt[DB_NAMELEN];
	char tmp[DB_NAMELEN];
	if(strlen(dbname)>DB_NAMELEN-1)
		return(FALSE);
	strncpy(txt,dbname,DB_NAMELEN-1);
	txt[DB_NAMELEN-1]=0;
        if(sscanf(txt,"%[_a-zA-Z]%[^_a-zA-Z0-9]",tmp,tmp)==1)
		return(TRUE);
        if(sscanf(txt,"%[_a-zA-Z]%[_a-zA-Z0-9]%[^_a-zA-Z0-9]",tmp,tmp,tmp)==2)
		return(TRUE);
	return(FALSE);
}


int is_pg_logname(char *username){
	if(strlen(username)>LOGNAMELEN-1)
		return(FALSE);
	return(TRUE);
}

