/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.db.tools.validate;

import org.apache.iotdb.commons.consensus.index.impl.MinimumProgressIndex;
import org.apache.iotdb.db.storageengine.dataregion.tsfile.TsFileResource;

import org.apache.tsfile.common.constant.TsFileConstant;
import org.slf4j.Logger;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

public class TsFileResourcePipeStatisticsSetTool {

  private static final Logger LOGGER =
      org.slf4j.LoggerFactory.getLogger(TsFileResourcePipeStatisticsSetTool.class);

  private static final String USAGE =
      "Usage: [--isGeneratedByPipe true|false] [--resetProgressIndex] --dirs <dir1> <dir2> ...\n"
          + "  --isGeneratedByPipe: whether the TsFileResource is isGeneratedByPipe to be generated by pipe\n"
          + "  --resetProgressIndex: whether to reset the TsFileResources' progressIndexes\n"
          + "  --dirs: list of data directories to validate and repair";

  private static final Set<File> dataDirs = new ConcurrentSkipListSet<>();
  private static AtomicBoolean isGeneratedByPipeMark = null;
  private static boolean resetProgressIndex = false;

  private static final AtomicLong runtime = new AtomicLong(System.currentTimeMillis());

  private static final AtomicInteger totalTsFileNum = new AtomicInteger(0);
  private static final AtomicInteger toResetFlagNum = new AtomicInteger(0);
  private static final AtomicInteger toResetProgressIndexNum = new AtomicInteger(0);
  private static final AtomicInteger changedNum = new AtomicInteger(0);

  // Usage: [--isGeneratedByPipe true|false] [--resetProgressIndex] --dirs <dir1> <dir2> ...
  public static void main(String[] args) throws IOException {
    parseCommandLineArgs(args);
    final List<File> partitionDirs = findAllPartitionDirs();
    partitionDirs.parallelStream()
        .forEach(TsFileResourcePipeStatisticsSetTool::validateAndRepairTsFileResourcesInPartition);
    printStatistics();
  }

  private static void parseCommandLineArgs(final String[] args) {
    final Set<String> argSet =
        new ConcurrentSkipListSet<>(
            args.length > 0 ? Arrays.asList(args) : Collections.emptyList());
    if (args.length == 0
        || argSet.contains("--help")
        || argSet.contains("-h")
        || !((argSet.contains("--isGeneratedByPipe") || argSet.contains("--resetProgressIndex"))
            && argSet.contains("--dirs"))) {
      LOGGER.info(USAGE);
      System.exit(1);
    }

    for (int i = 0; i < args.length; i++) {
      if ("--isGeneratedByPipe".equals(args[i]) && i + 1 < args.length) {
        isGeneratedByPipeMark = new AtomicBoolean(Boolean.parseBoolean(args[++i]));
      } else if ("--resetProgressIndex".equals(args[i])) {
        resetProgressIndex = true;
      } else if ("--dirs".equals(args[i]) && i + 1 < args.length) {
        i++;
        while (i < args.length && !args[i].startsWith("--")) {
          dataDirs.add(new File(args[i++]));
        }
        i--;
      } else {
        LOGGER.info("Unknown argument: {}", args[i]);
        LOGGER.info(USAGE);
        // Exit if an unknown argument is encountered
        System.exit(1);
      }
    }

    if (dataDirs.isEmpty()) {
      LOGGER.info("No data directories provided. Please specify with --dirs <dir1> <dir2> ...");
      System.exit(1);
    }

    LOGGER.info("------------------------------------------------------");
    LOGGER.info("isGeneratedByPipe mark: {}", isGeneratedByPipeMark);
    LOGGER.info("resetProgressIndex: {}", resetProgressIndex);
    LOGGER.info("Data directories: ");
    for (File dir : dataDirs) {
      LOGGER.info("  {}", dir.getAbsolutePath());
    }
    LOGGER.info("------------------------------------------------------");
  }

  private static List<File> findAllPartitionDirs() {
    final List<File> partitionDirs = new ArrayList<>();
    for (final File dataDir : dataDirs) {
      if (dataDir.exists() && dataDir.isDirectory()) {
        partitionDirs.addAll(findLeafDirectories(dataDir));
      }
    }
    return partitionDirs;
  }

  public static List<File> findLeafDirectories(File dir) {
    List<File> leafDirectories = new ArrayList<>();

    File[] files = dir.listFiles();

    if (files == null || files.length == 0) {
      leafDirectories.add(dir);
      return leafDirectories;
    }

    for (File file : files) {
      if (file.isDirectory()) {
        leafDirectories.addAll(findLeafDirectories(file));
      }
    }

    if (leafDirectories.isEmpty()) {
      leafDirectories.add(dir);
    }

    return leafDirectories;
  }

  private static void validateAndRepairTsFileResourcesInPartition(final File partitionDir) {
    final AtomicInteger totalTsFileResource = new AtomicInteger(0);
    final AtomicInteger toResetFlagResource = new AtomicInteger(0);
    final AtomicInteger toResetProgressIndexResource = new AtomicInteger(0);
    final AtomicInteger changedResource = new AtomicInteger(0);

    try {
      final List<TsFileResource> resources =
          loadAllTsFileResources(Collections.singletonList(partitionDir));
      totalTsFileResource.addAndGet(resources.size());

      for (final TsFileResource resource : resources) {
        try {
          validateAndRepairSingleTsFileResource(
              resource, toResetFlagResource, toResetProgressIndexResource, changedResource);
        } catch (final Exception e) {
          // Continue processing other resources even if one fails
          LOGGER.warn(
              "Error validating or repairing resource {}: {}",
              resource.getTsFile().getAbsolutePath(),
              e.getMessage(),
              e);
        }
      }
    } catch (final Exception e) {
      LOGGER.warn(
          "Error loading resources from partition {}: {}",
          partitionDir.getAbsolutePath(),
          e.getMessage(),
          e);
    }

    totalTsFileNum.addAndGet(totalTsFileResource.get());
    toResetFlagNum.addAndGet(toResetFlagResource.get());
    toResetProgressIndexNum.addAndGet(toResetProgressIndexResource.get());
    changedNum.addAndGet(changedResource.get());
    LOGGER.info(
        "TimePartition {} has {} total resources, {} to set isGeneratedByPipe resources, {} to reset progressIndex resources, {} changed resources. Process completed.",
        partitionDir,
        totalTsFileResource.get(),
        toResetFlagResource.get(),
        toResetProgressIndexResource.get(),
        changedResource.get());
  }

  private static List<TsFileResource> loadAllTsFileResources(List<File> timePartitionDirs)
      throws IOException {
    final List<TsFileResource> resources = new ArrayList<>();

    for (final File timePartitionDir : timePartitionDirs) {
      for (final File tsfile : Objects.requireNonNull(timePartitionDir.listFiles())) {
        final String filePath = tsfile.getAbsolutePath();
        if (!filePath.endsWith(TsFileConstant.TSFILE_SUFFIX) || !tsfile.isFile()) {
          continue;
        }
        String resourcePath = tsfile.getAbsolutePath() + TsFileResource.RESOURCE_SUFFIX;

        if (!new File(resourcePath).exists()) {
          LOGGER.info(
              "{} is skipped because resource file is not exist.", tsfile.getAbsolutePath());
          continue;
        }

        TsFileResource resource = new TsFileResource(tsfile);
        resource.deserialize();
        resource.close();
        resources.add(resource);
      }
    }

    return resources;
  }

  /**
   * Validates and repairs a single TsFileResource.
   *
   * @param resource the TsFileResource to validate and repair
   * @return true if the resource needs to be repaired and false if it is valid
   */
  private static void validateAndRepairSingleTsFileResource(
      final TsFileResource resource,
      final AtomicInteger toResetFlagResource,
      final AtomicInteger toResetProgressIndexResource,
      final AtomicInteger changedResource) {
    boolean skip = true;
    if (Objects.nonNull(isGeneratedByPipeMark)
        && resource.isGeneratedByPipe() != isGeneratedByPipeMark.get()) {
      // The resource is valid, no need to repair
      LOGGER.info(
          "Repairing TsFileResource: {}, isGeneratedByPipe mark: {}, actual mark: {}",
          resource.getTsFile().getAbsolutePath(),
          isGeneratedByPipeMark.get(),
          resource.isGeneratedByPipe());

      toResetFlagResource.getAndIncrement();
      skip = false;
    }

    if (resetProgressIndex && resource.getProgressIndex() != MinimumProgressIndex.INSTANCE) {
      // The resource is valid, no need to repair
      LOGGER.info(
          "Resetting TsFileResource:{} 's progressIndex to minimum, original progressIndex: {}",
          resource.getTsFile().getAbsolutePath(),
          resource.getProgressIndex());

      toResetProgressIndexResource.getAndIncrement();
      skip = false;
    }
    if (skip) {
      return;
    }
    changedResource.getAndIncrement();

    try {
      repairSingleTsFileResource(resource);

      LOGGER.info(
          "Marked TsFileResource as {} in resource: {}",
          isGeneratedByPipeMark.get(),
          resource.getTsFile().getAbsolutePath());
      LOGGER.info(
          "Reset TsFileResource:{} 's progressIndex to minimum.",
          resource.getTsFile().getAbsolutePath());
    } catch (final Exception e) {
      LOGGER.warn(
          "ERROR: Failed to repair TsFileResource: {}", resource.getTsFile().getAbsolutePath(), e);
    }
  }

  private static void repairSingleTsFileResource(TsFileResource resource) throws IOException {
    if (Objects.nonNull(isGeneratedByPipeMark)) {
      resource.setGeneratedByPipe(isGeneratedByPipeMark.get());
    }
    if (resetProgressIndex) {
      resource.setProgressIndex(MinimumProgressIndex.INSTANCE);
    }
    resource.serialize();
  }

  private static void printStatistics() {
    LOGGER.info("------------------------------------------------------");
    LOGGER.info("Validation and repair completed. Statistics:");
    LOGGER.info(
        "Total time taken: {} ms, total TsFile resources: {}, set isGeneratedByPipe resources: {}, reset progressIndex resources: {}, changed resources: {}",
        System.currentTimeMillis() - runtime.get(),
        totalTsFileNum.get(),
        toResetFlagNum.get(),
        toResetProgressIndexNum.get(),
        changedNum.get());
  }
}
