/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

use crate::args::common::ListModeExt;
use clap::Args;
use iggy::cli::utils::login_session_expiry::LoginSessionExpiry;
use iggy::snapshot::{SnapshotCompression, SystemSnapshotType};

#[derive(Debug, Clone, Args)]
pub(crate) struct PingArgs {
    /// Stop after sending count Ping packets
    #[arg(short, long, default_value_t = 1)]
    pub(crate) count: u32,
}

#[derive(Debug, Clone, Args)]
pub(crate) struct LoginArgs {
    /// Login session expiry time in human-readable format
    ///
    /// Expiry time must be expressed in human-readable format like 1hour 15min 2s.
    /// If not set default value 15minutes is used. Using "none" disables session expiry time.
    #[clap(verbatim_doc_comment)]
    #[arg(value_parser = clap::value_parser!(LoginSessionExpiry), group = "store")]
    pub(crate) expiry: Option<Vec<LoginSessionExpiry>>,
}

#[derive(Debug, Clone, Args)]
pub(crate) struct StatsArgs {
    /// List mode (table, list, JSON, TOML)
    #[clap(short, long, value_enum, default_value_t = ListModeExt::Table)]
    pub(crate) output: ListModeExt,
}

#[derive(Debug, Clone, Args)]
pub(crate) struct SnapshotArgs {
    /// Specify snapshot compression method.
    ///
    /// Available options:
    ///
    /// - `stored`: No compression
    /// - `deflated`: Standard deflate compression
    /// - `bzip2`: Higher compression ratio but slower
    /// - `zstd`: Fast compression and decompression
    /// - `lzma`: High compression, suitable for large files
    /// - `xz`: Similar to `lzma` but often faster in decompression
    ///
    /// Examples:
    /// - `--compression bzip2` for higher compression.
    /// - `--compression none` to store without compression.
    #[arg(verbatim_doc_comment, short, long, value_parser = clap::value_parser!(SnapshotCompression))]
    pub(crate) compression: Option<SnapshotCompression>,

    /// Specify types of snapshots to include.
    ///
    /// Available snapshot types:
    /// - `filesystem_overview`: Provides an overview of the filesystem structure.
    /// - `process_list`: Captures the list of active processes.
    /// - `resource_usage`: Monitors CPU, memory, and other system resources.
    /// - `test`: Used for testing purposes.
    /// - `server_logs`: Server logs from the specified logging directory, useful for system diagnostics.
    /// - `server_config`: Server configuration.
    /// - `all`: Take all available snapshots.
    ///
    /// Examples:
    /// - `--snapshot-types filesystem_overview process_list`
    /// - `--snapshot-types resource_usage`
    #[arg(verbatim_doc_comment, short, long, value_parser = clap::value_parser!(SystemSnapshotType), value_delimiter = ' ', num_args = 1..)]
    pub(crate) snapshot_types: Option<Vec<SystemSnapshotType>>,

    /// Define the output directory for the snapshot file.
    ///
    /// This directory will contain the snapshot files generated by the command.
    ///
    /// Examples:
    /// - `--out-dir /var/snapshots`
    /// - `--out-dir ./snapshots`
    #[arg(verbatim_doc_comment, short, long)]
    pub(crate) out_dir: Option<String>,
}
