/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.gravitino.job.local;

import java.io.File;
import java.util.Map;
import org.apache.gravitino.job.JobTemplate;
import org.apache.gravitino.job.ShellJobTemplate;
import org.apache.gravitino.job.SparkJobTemplate;

public abstract class LocalProcessBuilder {

  protected final JobTemplate jobTemplate;

  protected final File workingDirectory;

  protected LocalProcessBuilder(JobTemplate jobTemplate, Map<String, String> configs) {
    this.jobTemplate = jobTemplate;
    // Executable should be in the working directory, so we can figure out the working directory
    // from the executable path.
    this.workingDirectory = new File(jobTemplate.executable()).getAbsoluteFile().getParentFile();
  }

  public abstract Process start();

  public static LocalProcessBuilder create(JobTemplate jobTemplate, Map<String, String> configs) {
    if (jobTemplate instanceof ShellJobTemplate) {
      return new ShellProcessBuilder((ShellJobTemplate) jobTemplate, configs);
    } else if (jobTemplate instanceof SparkJobTemplate) {
      return new SparkProcessBuilder((SparkJobTemplate) jobTemplate, configs);
    } else {
      throw new IllegalArgumentException(
          "Unsupported job template type: " + jobTemplate.getClass().getName());
    }
  }
}
