/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;

/**
 * Specifies how fields from a completion item should be combined with those
 * from {@link CompletionList#itemDefaults}.
 * <p>
 * If unspecified, all fields will be treated as {@link ApplyKind#Replace}.
 * <p>
 * If a field's value is {@link ApplyKind#Replace}, the value from a completion item
 * (if provided and not {@code null}) will always be used instead of the value
 * from {@link CompletionList#itemDefaults}.
 * <p>
 * If a field's value is {@link ApplyKind#Merge}, the values will be merged using
 * the rules defined against each field in {@link CompletionApplyKind}.
 * <p>
 * Servers are only allowed to return {@link CompletionList#applyKind} if the client
 * signals support for this via the {@link CompletionListCapabilities#applyKindSupport}
 * capability.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class CompletionApplyKind {
  /**
   * Specifies whether {@link CompletionItem#commitCharacters commitCharacters}
   * on a completion will replace or be merged with those in
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Replace}, the commit characters from the completion item
   * will always be used unless not provided, in which case those from
   * {@link CompletionItemDefaults#commitCharacters} will be used. An
   * empty list can be used if a completion item does not have any commit
   * characters and also should not use those from
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Merge}, the commit characters for the completion will be
   * the union of all values in both {@link CompletionItemDefaults#commitCharacters}
   * and the completion's own {@link CompletionItem#commitCharacters commitCharacters}.
   */
  private ApplyKind commitCharacters;

  /**
   * Specifies whether the {@link CompletionItem#data data} field on a completion
   * will replace or be merged with data from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Replace}, the data from the completion item will be used
   * if provided (and not {@code null}), otherwise {@link CompletionItemDefaults#data}
   * will be used. An empty object can be used if a completion item does not have
   * any data but also should not use the value from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Merge}, a shallow merge will be performed between
   * {@link CompletionItemDefaults#data} and the completion's own data
   * using the following rules:
   * <ul>
   * <li>If a completion's {@link CompletionItem#data data} field is not provided
   * (or {@code null}), the entire {@link CompletionItemDefaults#data} field will be
   * used as-is.
   * <li>If a completion's {@link CompletionItem#data data} field is provided,
   * each field will overwrite the field of the same name in
   * {@link CompletionItemDefaults#data}, but no merging of nested fields
   * within that value will occur.
   * </ul>
   */
  private ApplyKind data;

  /**
   * Specifies whether {@link CompletionItem#commitCharacters commitCharacters}
   * on a completion will replace or be merged with those in
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Replace}, the commit characters from the completion item
   * will always be used unless not provided, in which case those from
   * {@link CompletionItemDefaults#commitCharacters} will be used. An
   * empty list can be used if a completion item does not have any commit
   * characters and also should not use those from
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Merge}, the commit characters for the completion will be
   * the union of all values in both {@link CompletionItemDefaults#commitCharacters}
   * and the completion's own {@link CompletionItem#commitCharacters commitCharacters}.
   */
  public ApplyKind getCommitCharacters() {
    return this.commitCharacters;
  }

  /**
   * Specifies whether {@link CompletionItem#commitCharacters commitCharacters}
   * on a completion will replace or be merged with those in
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Replace}, the commit characters from the completion item
   * will always be used unless not provided, in which case those from
   * {@link CompletionItemDefaults#commitCharacters} will be used. An
   * empty list can be used if a completion item does not have any commit
   * characters and also should not use those from
   * {@link CompletionItemDefaults#commitCharacters}.
   * <p>
   * If {@link ApplyKind#Merge}, the commit characters for the completion will be
   * the union of all values in both {@link CompletionItemDefaults#commitCharacters}
   * and the completion's own {@link CompletionItem#commitCharacters commitCharacters}.
   */
  public void setCommitCharacters(final ApplyKind commitCharacters) {
    this.commitCharacters = commitCharacters;
  }

  /**
   * Specifies whether the {@link CompletionItem#data data} field on a completion
   * will replace or be merged with data from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Replace}, the data from the completion item will be used
   * if provided (and not {@code null}), otherwise {@link CompletionItemDefaults#data}
   * will be used. An empty object can be used if a completion item does not have
   * any data but also should not use the value from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Merge}, a shallow merge will be performed between
   * {@link CompletionItemDefaults#data} and the completion's own data
   * using the following rules:
   * <ul>
   * <li>If a completion's {@link CompletionItem#data data} field is not provided
   * (or {@code null}), the entire {@link CompletionItemDefaults#data} field will be
   * used as-is.
   * <li>If a completion's {@link CompletionItem#data data} field is provided,
   * each field will overwrite the field of the same name in
   * {@link CompletionItemDefaults#data}, but no merging of nested fields
   * within that value will occur.
   * </ul>
   */
  public ApplyKind getData() {
    return this.data;
  }

  /**
   * Specifies whether the {@link CompletionItem#data data} field on a completion
   * will replace or be merged with data from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Replace}, the data from the completion item will be used
   * if provided (and not {@code null}), otherwise {@link CompletionItemDefaults#data}
   * will be used. An empty object can be used if a completion item does not have
   * any data but also should not use the value from {@link CompletionItemDefaults#data}.
   * <p>
   * If {@link ApplyKind#Merge}, a shallow merge will be performed between
   * {@link CompletionItemDefaults#data} and the completion's own data
   * using the following rules:
   * <ul>
   * <li>If a completion's {@link CompletionItem#data data} field is not provided
   * (or {@code null}), the entire {@link CompletionItemDefaults#data} field will be
   * used as-is.
   * <li>If a completion's {@link CompletionItem#data data} field is provided,
   * each field will overwrite the field of the same name in
   * {@link CompletionItemDefaults#data}, but no merging of nested fields
   * within that value will occur.
   * </ul>
   */
  public void setData(final ApplyKind data) {
    this.data = data;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("commitCharacters", this.commitCharacters);
    b.add("data", this.data);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CompletionApplyKind other = (CompletionApplyKind) obj;
    if (this.commitCharacters == null) {
      if (other.commitCharacters != null)
        return false;
    } else if (!this.commitCharacters.equals(other.commitCharacters))
      return false;
    if (this.data == null) {
      if (other.data != null)
        return false;
    } else if (!this.data.equals(other.data))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.commitCharacters== null) ? 0 : this.commitCharacters.hashCode());
    return prime * result + ((this.data== null) ? 0 : this.data.hashCode());
  }
}
