//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.common;

import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Sets.set;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.eclipse.escet.cif.metamodel.cif.InputParameter;
import org.eclipse.escet.cif.metamodel.cif.automata.Assignment;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.ElifUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.IfUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.declarations.Declaration;
import org.eclipse.escet.cif.metamodel.cif.expressions.CompInstWrapExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.CompParamWrapExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ContVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.InputVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ProjectionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TupleExpression;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.Lists;

/** CIF addressables utility methods. */
public class CifAddressableUtils {
    /** Constructor for the {@link CifAddressableUtils} class. */
    private CifAddressableUtils() {
        // Static class.
    }

    /**
     * Returns the addressable variable referred to by the given addressable expression.
     *
     * @param addr The addressable expression. It must refer to a discrete variable, continuous variable or input
     *     variable addressable. So, no tuples, projections, input parameters, component instantiation wrapping
     *     expressions, and component parameter wrapping expressions.
     * @return The discrete variable, continuous variable, input variable, function local variable or function
     *     parameter.
     */
    public static Declaration getVariable(Expression addr) {
        return switch (addr) {
            case DiscVariableExpression daddr -> daddr.getVariable();
            case ContVariableExpression caddr -> caddr.getVariable();
            case InputVariableExpression iaddr -> {
                if (iaddr.getVariable().eContainer() instanceof InputParameter) {
                    throw new AssertionError("Unsupported input parameter addressable: " + addr);
                } else {
                    yield iaddr.getVariable();
                }
            }
            default -> throw new AssertionError("Unexpected addressable: " + addr);
        };
    }

    /**
     * Returns the variable reference expressions for the variables (partially) assigned in an addressable, by
     * recursively stripping of tuple expressions and projection expressions.
     *
     * <p>
     * This method does not support component instantiation wrapping expressions and component parameter wrapping
     * expressions, which can only occur for assignments to input variables as part of SVG input mappings with updates,
     * in specifications with component definitions/instantiations.
     * </p>
     *
     * @param addr The addressable expression.
     * @return The variable reference expressions for the variables (partially) assigned in the addressable.
     */
    public static List<Expression> getRefExprs(Expression addr) {
        return switch (addr) {
            case TupleExpression tupleAddr -> {
                // Recursively strip off top level tuples.
                List<Expression> rslt = list();
                for (Expression elem: tupleAddr.getFields()) {
                    rslt.addAll(getRefExprs(elem));
                }
                yield rslt;
            }
            case ProjectionExpression projAddr -> {
                // Strip off top level projections, but not anything else.
                while (addr instanceof ProjectionExpression p) {
                    addr = p.getChild();
                }
                yield list(addr);
            }
            case CompInstWrapExpression wrap -> throw new AssertionError("Unsupported addressable: " + wrap);
            case CompParamWrapExpression wrap -> throw new AssertionError("Unsupported addressable: " + wrap);
            default -> list(addr); // Direct variable reference (if the addressable is valid to begin with).
        };
    }

    /**
     * Returns the variables (partially) assigned in an addressable, by recursively stripping of tuple expressions and
     * projection expressions, and obtaining the variables from the variable reference expressions.
     *
     * <p>
     * This method does not support input parameters, component instantiation wrapping expressions and component
     * parameter wrapping expressions, which can only occur for assignments to input variables as part of SVG input
     * mappings with updates, in specifications with component definitions/instantiations.
     * </p>
     *
     * <p>
     * In the CIF language, it is allowed to assign different parts of the same variable, in different assignments on a
     * single edge, or even in a single (multi-)assignment. Similarly, this applies to multiple updates in an SVG input
     * mapping, and multi-assignments in functions. This is allowed in CIF, as long as it can be statically checked that
     * there is never a possibility for overlap (assigning the same part of the variable more than once). This method
     * however, assumes that each variable that is fully or partially assigned in a single (multi-)assignment, is for a
     * different variable altogether.
     * </p>
     *
     * <p>
     * Due to the above assumption, this method should be avoided. Use the {@link #collectAddrVars(Expression, Set)}
     * method instead, if possible.
     * </p>
     *
     * <p>
     * Note that even though the result is a set, all addressable variables are included. Due to the above assumption
     * that all variables are in the addressable are unique, the number of variables in the set is equal to the number
     * of (partial) variable assignments. The set allows for fast element testing.
     * </p>
     *
     * <p>
     * The resulting set can be iterated, to return the addressable variables in the order that they occur (from left to
     * right) in the textual syntax of the addressable.
     * </p>
     *
     * @param addr The addressable expression.
     * @return The (partially) assigned variables.
     * @throws DuplVarAsgnException If the assumption that all variables that are (partially) assigned are unique,
     *     doesn't hold.
     */
    public static Set<Declaration> getRefs(Expression addr) throws DuplVarAsgnException {
        List<Expression> refExprs = getRefExprs(addr);
        Set<Declaration> rslt = set();
        for (Expression refExpr: refExprs) {
            rslt.add(switch (refExpr) {
                case DiscVariableExpression discRef -> discRef.getVariable();
                case ContVariableExpression contRef -> contRef.getVariable();
                case InputVariableExpression inputRef -> {
                    if (inputRef.getVariable().eContainer() instanceof InputParameter) {
                        throw new AssertionError("Unsupported input parameter addressable: " + addr);
                    } else {
                        yield inputRef.getVariable();
                    }
                }
                default -> throw new AssertionError("Unexpected addressable: " + addr);
            });
        }
        if (refExprs.size() != rslt.size()) {
            throw new DuplVarAsgnException();
        }
        return rslt;
    }

    /**
     * Exception indicating duplicate assignment to (a part of) a variable, violating the 'unique assigned variables'
     * assumption of the {@link #getRefs} method.
     */
    public static class DuplVarAsgnException extends Exception {
        // No message or cause for this exception.
    }

    /**
     * Does the given addressable (recursively) have any projections?
     *
     * @param addr The addressable.
     * @return {@code true} if it has any projections, {@code false} otherwise.
     */
    public static boolean hasProjs(Expression addr) {
        return switch (addr) {
            case TupleExpression tupleAddr -> tupleAddr.getFields().stream().anyMatch(f -> hasProjs(f));
            case ProjectionExpression projAddr -> true;
            case DiscVariableExpression discAddr -> false;
            case ContVariableExpression contAddr -> false;
            case InputVariableExpression inputAddr -> false;
            case CompInstWrapExpression compInstWrapAddr -> hasProjs(compInstWrapAddr.getReference());
            case CompParamWrapExpression compParamWrapAddr -> hasProjs(compParamWrapAddr.getReference());
            default -> throw new AssertionError("Unknown addressable: " + addr);
        };
    }

    /**
     * Collect the projection addressables in the given addressable (recursively). A projection is only collected if it
     * represents a projection addressable; no projections are collected within projection index expressions.
     *
     * @param <T> The type of the collection.
     * @param addr The addressable.
     * @param collected The collection in which to collect the projections. Is extended in-place.
     * @return The collection.
     */
    public static <T extends Collection<ProjectionExpression>> T collectProjs(Expression addr, T collected) {
        if (addr instanceof TupleExpression tupleAddr) {
            for (Expression field: tupleAddr.getFields()) {
                collectProjs(field, collected);
            }
            return collected;
        } else if (addr instanceof ProjectionExpression projExpr) {
            collectProjs(projExpr.getChild(), collected);
            collected.add(projExpr);
            return collected;
        } else if (addr instanceof DiscVariableExpression) {
            return collected;
        } else if (addr instanceof ContVariableExpression) {
            return collected;
        } else if (addr instanceof InputVariableExpression) {
            return collected;
        } else if (addr instanceof CompInstWrapExpression compInstWrapAddr) {
            collectProjs(compInstWrapAddr.getReference(), collected);
            return collected;
        } else if (addr instanceof CompParamWrapExpression compParamWrapAddr) {
            collectProjs(compParamWrapAddr.getReference(), collected);
            return collected;
        } else {
            throw new AssertionError("Unknown addressable: " + addr);
        }
    }

    /**
     * Strips the projections of a possibly projected addressable variable.
     *
     * @param addr The possibly projected addressable variable.
     * @return The addressable variable, without any projections.
     */
    public static Expression stripProjs(Expression addr) {
        Expression rslt = addr;
        while (rslt instanceof ProjectionExpression projExpr) {
            rslt = projExpr.getChild();
        }
        return rslt;
    }

    /**
     * Collects the projection expressions in the addressable. Projections closest to the addressable variable are more
     * to the beginning of the resulting list than projections farther away from the addressable variable.
     *
     * @param addr The addressable. Must not be a tuple addressable, component instantiation wrapping expressions, or
     *     component parameter wrapping expressions.
     * @return The projection expressions.
     */
    public static List<ProjectionExpression> collectProjs(Expression addr) {
        Assert.check(!(addr instanceof TupleExpression));
        Assert.check(!(addr instanceof CompInstWrapExpression));
        Assert.check(!(addr instanceof CompParamWrapExpression));

        // Collect projects while we strip them off.
        List<ProjectionExpression> rslt = list();
        while (addr instanceof ProjectionExpression) {
            ProjectionExpression paddr = (ProjectionExpression)addr;
            rslt.add(paddr);
            addr = paddr.getChild();
        }

        // Reverse the list.
        rslt = Lists.reverse(rslt);

        // Return the collected projections.
        return rslt;
    }

    /**
     * Collects the variables (partially) assigned in the given updates.
     *
     * @param updates The updates. Any assignments (recursively) present in the updates must (recursively) not have any
     *     input parameters, component instantiation wrapping expressions or component parameter wrapping expressions in
     *     their addressables.
     * @param vars The variables collected so far. Is modified in-place.
     * @see #getRefs
     */
    public static void collectAddrVars(List<Update> updates, Set<Declaration> vars) {
        for (Update update: updates) {
            collectAddrVars(update, vars);
        }
    }

    /**
     * Collects the variables (partially) assigned in the given update.
     *
     * @param update The update. Any assignments (recursively) present in the updates must (recursively) not have any
     *     input parameters component instantiation wrapping expressions or component parameter wrapping expressions in
     *     their addressables.
     * @param vars The variables collected so far. Is modified in-place.
     */
    public static void collectAddrVars(Update update, Set<Declaration> vars) {
        if (update instanceof IfUpdate) {
            IfUpdate ifUpd = (IfUpdate)update;
            collectAddrVars(ifUpd.getThens(), vars);
            for (ElifUpdate elifUpd: ifUpd.getElifs()) {
                collectAddrVars(elifUpd.getThens(), vars);
            }
            collectAddrVars(ifUpd.getElses(), vars);
        } else {
            Assignment asgn = (Assignment)update;
            collectAddrVars(asgn.getAddressable(), vars);
        }
    }

    /**
     * Collects the variables (partially) assigned in the given addressable.
     *
     * @param addr The addressable. Must not be, nor recursively include, any input parameters component instantiation
     *     wrapping expressions or component parameter wrapping expressions.
     * @param vars The variables collected so far. Is modified in-place.
     */
    public static void collectAddrVars(Expression addr, Set<Declaration> vars) {
        if (addr instanceof TupleExpression tupleAddr) {
            for (Expression elem: tupleAddr.getFields()) {
                collectAddrVars(elem, vars);
            }
        } else if (addr instanceof ProjectionExpression) {
            while (addr instanceof ProjectionExpression p) {
                addr = p.getChild();
            }
            collectAddrVars(addr, vars);
        } else if (addr instanceof DiscVariableExpression discRef) {
            vars.add(discRef.getVariable());
        } else if (addr instanceof ContVariableExpression contRef) {
            vars.add(contRef.getVariable());
        } else if (addr instanceof InputVariableExpression inputRef) {
            if (inputRef.getVariable().eContainer() instanceof InputParameter) {
                throw new RuntimeException("Unsupported input parameter addressable: " + addr);
            } else {
                vars.add(inputRef.getVariable());
            }
        } else {
            throw new RuntimeException("Unknown addr: " + addr);
        }
    }

    /**
     * Collects the variables (partially) assigned in the given automaton.
     *
     * @param aut The automaton.
     * @param vars The variables collected so far. Is modified in-place.
     */
    public static void collectAddrVars(Automaton aut, Set<Declaration> vars) {
        // Note that updates on edges can't assign input parameters, nor can they have component instantiation or
        // component parameter wrapping expressions.
        for (Location loc: aut.getLocations()) {
            for (Edge edge: loc.getEdges()) {
                collectAddrVars(edge.getUpdates(), vars);
            }
        }
    }
}
