// license:BSD-3-Clause
// copyright-holders:Vas Crabb, Ryan Holtz
/***************************************************************************

    rgbutil.cpp

    Utility definitions for RGB manipulation. Allows RGB handling to be
    performed in an abstracted fashion and optimized with SIMD.

***************************************************************************/

#include "emu.h"
#include "rgbutil.h"

#if defined(__SSE2__) || defined(_M_X64) || (defined(_M_IX86_FP) && (_M_IX86_FP >= 2))

#include <emmintrin.h>

namespace {

struct
{
	__m128  dummy_for_alignment;
	s16     scale_table[256][8];
}
const rgbutil_statics =
{
	{ 0 },
	{
		{   0, 256,   0, 256,   0, 256,   0, 256 }, {   1, 255,   1, 255,   1, 255,   1, 255 },
		{   2, 254,   2, 254,   2, 254,   2, 254 }, {   3, 253,   3, 253,   3, 253,   3, 253 },
		{   4, 252,   4, 252,   4, 252,   4, 252 }, {   5, 251,   5, 251,   5, 251,   5, 251 },
		{   6, 250,   6, 250,   6, 250,   6, 250 }, {   7, 249,   7, 249,   7, 249,   7, 249 },
		{   8, 248,   8, 248,   8, 248,   8, 248 }, {   9, 247,   9, 247,   9, 247,   9, 247 },
		{  10, 246,  10, 246,  10, 246,  10, 246 }, {  11, 245,  11, 245,  11, 245,  11, 245 },
		{  12, 244,  12, 244,  12, 244,  12, 244 }, {  13, 243,  13, 243,  13, 243,  13, 243 },
		{  14, 242,  14, 242,  14, 242,  14, 242 }, {  15, 241,  15, 241,  15, 241,  15, 241 },
		{  16, 240,  16, 240,  16, 240,  16, 240 }, {  17, 239,  17, 239,  17, 239,  17, 239 },
		{  18, 238,  18, 238,  18, 238,  18, 238 }, {  19, 237,  19, 237,  19, 237,  19, 237 },
		{  20, 236,  20, 236,  20, 236,  20, 236 }, {  21, 235,  21, 235,  21, 235,  21, 235 },
		{  22, 234,  22, 234,  22, 234,  22, 234 }, {  23, 233,  23, 233,  23, 233,  23, 233 },
		{  24, 232,  24, 232,  24, 232,  24, 232 }, {  25, 231,  25, 231,  25, 231,  25, 231 },
		{  26, 230,  26, 230,  26, 230,  26, 230 }, {  27, 229,  27, 229,  27, 229,  27, 229 },
		{  28, 228,  28, 228,  28, 228,  28, 228 }, {  29, 227,  29, 227,  29, 227,  29, 227 },
		{  30, 226,  30, 226,  30, 226,  30, 226 }, {  31, 225,  31, 225,  31, 225,  31, 225 },
		{  32, 224,  32, 224,  32, 224,  32, 224 }, {  33, 223,  33, 223,  33, 223,  33, 223 },
		{  34, 222,  34, 222,  34, 222,  34, 222 }, {  35, 221,  35, 221,  35, 221,  35, 221 },
		{  36, 220,  36, 220,  36, 220,  36, 220 }, {  37, 219,  37, 219,  37, 219,  37, 219 },
		{  38, 218,  38, 218,  38, 218,  38, 218 }, {  39, 217,  39, 217,  39, 217,  39, 217 },
		{  40, 216,  40, 216,  40, 216,  40, 216 }, {  41, 215,  41, 215,  41, 215,  41, 215 },
		{  42, 214,  42, 214,  42, 214,  42, 214 }, {  43, 213,  43, 213,  43, 213,  43, 213 },
		{  44, 212,  44, 212,  44, 212,  44, 212 }, {  45, 211,  45, 211,  45, 211,  45, 211 },
		{  46, 210,  46, 210,  46, 210,  46, 210 }, {  47, 209,  47, 209,  47, 209,  47, 209 },
		{  48, 208,  48, 208,  48, 208,  48, 208 }, {  49, 207,  49, 207,  49, 207,  49, 207 },
		{  50, 206,  50, 206,  50, 206,  50, 206 }, {  51, 205,  51, 205,  51, 205,  51, 205 },
		{  52, 204,  52, 204,  52, 204,  52, 204 }, {  53, 203,  53, 203,  53, 203,  53, 203 },
		{  54, 202,  54, 202,  54, 202,  54, 202 }, {  55, 201,  55, 201,  55, 201,  55, 201 },
		{  56, 200,  56, 200,  56, 200,  56, 200 }, {  57, 199,  57, 199,  57, 199,  57, 199 },
		{  58, 198,  58, 198,  58, 198,  58, 198 }, {  59, 197,  59, 197,  59, 197,  59, 197 },
		{  60, 196,  60, 196,  60, 196,  60, 196 }, {  61, 195,  61, 195,  61, 195,  61, 195 },
		{  62, 194,  62, 194,  62, 194,  62, 194 }, {  63, 193,  63, 193,  63, 193,  63, 193 },
		{  64, 192,  64, 192,  64, 192,  64, 192 }, {  65, 191,  65, 191,  65, 191,  65, 191 },
		{  66, 190,  66, 190,  66, 190,  66, 190 }, {  67, 189,  67, 189,  67, 189,  67, 189 },
		{  68, 188,  68, 188,  68, 188,  68, 188 }, {  69, 187,  69, 187,  69, 187,  69, 187 },
		{  70, 186,  70, 186,  70, 186,  70, 186 }, {  71, 185,  71, 185,  71, 185,  71, 185 },
		{  72, 184,  72, 184,  72, 184,  72, 184 }, {  73, 183,  73, 183,  73, 183,  73, 183 },
		{  74, 182,  74, 182,  74, 182,  74, 182 }, {  75, 181,  75, 181,  75, 181,  75, 181 },
		{  76, 180,  76, 180,  76, 180,  76, 180 }, {  77, 179,  77, 179,  77, 179,  77, 179 },
		{  78, 178,  78, 178,  78, 178,  78, 178 }, {  79, 177,  79, 177,  79, 177,  79, 177 },
		{  80, 176,  80, 176,  80, 176,  80, 176 }, {  81, 175,  81, 175,  81, 175,  81, 175 },
		{  82, 174,  82, 174,  82, 174,  82, 174 }, {  83, 173,  83, 173,  83, 173,  83, 173 },
		{  84, 172,  84, 172,  84, 172,  84, 172 }, {  85, 171,  85, 171,  85, 171,  85, 171 },
		{  86, 170,  86, 170,  86, 170,  86, 170 }, {  87, 169,  87, 169,  87, 169,  87, 169 },
		{  88, 168,  88, 168,  88, 168,  88, 168 }, {  89, 167,  89, 167,  89, 167,  89, 167 },
		{  90, 166,  90, 166,  90, 166,  90, 166 }, {  91, 165,  91, 165,  91, 165,  91, 165 },
		{  92, 164,  92, 164,  92, 164,  92, 164 }, {  93, 163,  93, 163,  93, 163,  93, 163 },
		{  94, 162,  94, 162,  94, 162,  94, 162 }, {  95, 161,  95, 161,  95, 161,  95, 161 },
		{  96, 160,  96, 160,  96, 160,  96, 160 }, {  97, 159,  97, 159,  97, 159,  97, 159 },
		{  98, 158,  98, 158,  98, 158,  98, 158 }, {  99, 157,  99, 157,  99, 157,  99, 157 },
		{ 100, 156, 100, 156, 100, 156, 100, 156 }, { 101, 155, 101, 155, 101, 155, 101, 155 },
		{ 102, 154, 102, 154, 102, 154, 102, 154 }, { 103, 153, 103, 153, 103, 153, 103, 153 },
		{ 104, 152, 104, 152, 104, 152, 104, 152 }, { 105, 151, 105, 151, 105, 151, 105, 151 },
		{ 106, 150, 106, 150, 106, 150, 106, 150 }, { 107, 149, 107, 149, 107, 149, 107, 149 },
		{ 108, 148, 108, 148, 108, 148, 108, 148 }, { 109, 147, 109, 147, 109, 147, 109, 147 },
		{ 110, 146, 110, 146, 110, 146, 110, 146 }, { 111, 145, 111, 145, 111, 145, 111, 145 },
		{ 112, 144, 112, 144, 112, 144, 112, 144 }, { 113, 143, 113, 143, 113, 143, 113, 143 },
		{ 114, 142, 114, 142, 114, 142, 114, 142 }, { 115, 141, 115, 141, 115, 141, 115, 141 },
		{ 116, 140, 116, 140, 116, 140, 116, 140 }, { 117, 139, 117, 139, 117, 139, 117, 139 },
		{ 118, 138, 118, 138, 118, 138, 118, 138 }, { 119, 137, 119, 137, 119, 137, 119, 137 },
		{ 120, 136, 120, 136, 120, 136, 120, 136 }, { 121, 135, 121, 135, 121, 135, 121, 135 },
		{ 122, 134, 122, 134, 122, 134, 122, 134 }, { 123, 133, 123, 133, 123, 133, 123, 133 },
		{ 124, 132, 124, 132, 124, 132, 124, 132 }, { 125, 131, 125, 131, 125, 131, 125, 131 },
		{ 126, 130, 126, 130, 126, 130, 126, 130 }, { 127, 129, 127, 129, 127, 129, 127, 129 },
		{ 128, 128, 128, 128, 128, 128, 128, 128 }, { 129, 127, 129, 127, 129, 127, 129, 127 },
		{ 130, 126, 130, 126, 130, 126, 130, 126 }, { 131, 125, 131, 125, 131, 125, 131, 125 },
		{ 132, 124, 132, 124, 132, 124, 132, 124 }, { 133, 123, 133, 123, 133, 123, 133, 123 },
		{ 134, 122, 134, 122, 134, 122, 134, 122 }, { 135, 121, 135, 121, 135, 121, 135, 121 },
		{ 136, 120, 136, 120, 136, 120, 136, 120 }, { 137, 119, 137, 119, 137, 119, 137, 119 },
		{ 138, 118, 138, 118, 138, 118, 138, 118 }, { 139, 117, 139, 117, 139, 117, 139, 117 },
		{ 140, 116, 140, 116, 140, 116, 140, 116 }, { 141, 115, 141, 115, 141, 115, 141, 115 },
		{ 142, 114, 142, 114, 142, 114, 142, 114 }, { 143, 113, 143, 113, 143, 113, 143, 113 },
		{ 144, 112, 144, 112, 144, 112, 144, 112 }, { 145, 111, 145, 111, 145, 111, 145, 111 },
		{ 146, 110, 146, 110, 146, 110, 146, 110 }, { 147, 109, 147, 109, 147, 109, 147, 109 },
		{ 148, 108, 148, 108, 148, 108, 148, 108 }, { 149, 107, 149, 107, 149, 107, 149, 107 },
		{ 150, 106, 150, 106, 150, 106, 150, 106 }, { 151, 105, 151, 105, 151, 105, 151, 105 },
		{ 152, 104, 152, 104, 152, 104, 152, 104 }, { 153, 103, 153, 103, 153, 103, 153, 103 },
		{ 154, 102, 154, 102, 154, 102, 154, 102 }, { 155, 101, 155, 101, 155, 101, 155, 101 },
		{ 156, 100, 156, 100, 156, 100, 156, 100 }, { 157,  99, 157,  99, 157,  99, 157,  99 },
		{ 158,  98, 158,  98, 158,  98, 158,  98 }, { 159,  97, 159,  97, 159,  97, 159,  97 },
		{ 160,  96, 160,  96, 160,  96, 160,  96 }, { 161,  95, 161,  95, 161,  95, 161,  95 },
		{ 162,  94, 162,  94, 162,  94, 162,  94 }, { 163,  93, 163,  93, 163,  93, 163,  93 },
		{ 164,  92, 164,  92, 164,  92, 164,  92 }, { 165,  91, 165,  91, 165,  91, 165,  91 },
		{ 166,  90, 166,  90, 166,  90, 166,  90 }, { 167,  89, 167,  89, 167,  89, 167,  89 },
		{ 168,  88, 168,  88, 168,  88, 168,  88 }, { 169,  87, 169,  87, 169,  87, 169,  87 },
		{ 170,  86, 170,  86, 170,  86, 170,  86 }, { 171,  85, 171,  85, 171,  85, 171,  85 },
		{ 172,  84, 172,  84, 172,  84, 172,  84 }, { 173,  83, 173,  83, 173,  83, 173,  83 },
		{ 174,  82, 174,  82, 174,  82, 174,  82 }, { 175,  81, 175,  81, 175,  81, 175,  81 },
		{ 176,  80, 176,  80, 176,  80, 176,  80 }, { 177,  79, 177,  79, 177,  79, 177,  79 },
		{ 178,  78, 178,  78, 178,  78, 178,  78 }, { 179,  77, 179,  77, 179,  77, 179,  77 },
		{ 180,  76, 180,  76, 180,  76, 180,  76 }, { 181,  75, 181,  75, 181,  75, 181,  75 },
		{ 182,  74, 182,  74, 182,  74, 182,  74 }, { 183,  73, 183,  73, 183,  73, 183,  73 },
		{ 184,  72, 184,  72, 184,  72, 184,  72 }, { 185,  71, 185,  71, 185,  71, 185,  71 },
		{ 186,  70, 186,  70, 186,  70, 186,  70 }, { 187,  69, 187,  69, 187,  69, 187,  69 },
		{ 188,  68, 188,  68, 188,  68, 188,  68 }, { 189,  67, 189,  67, 189,  67, 189,  67 },
		{ 190,  66, 190,  66, 190,  66, 190,  66 }, { 191,  65, 191,  65, 191,  65, 191,  65 },
		{ 192,  64, 192,  64, 192,  64, 192,  64 }, { 193,  63, 193,  63, 193,  63, 193,  63 },
		{ 194,  62, 194,  62, 194,  62, 194,  62 }, { 195,  61, 195,  61, 195,  61, 195,  61 },
		{ 196,  60, 196,  60, 196,  60, 196,  60 }, { 197,  59, 197,  59, 197,  59, 197,  59 },
		{ 198,  58, 198,  58, 198,  58, 198,  58 }, { 199,  57, 199,  57, 199,  57, 199,  57 },
		{ 200,  56, 200,  56, 200,  56, 200,  56 }, { 201,  55, 201,  55, 201,  55, 201,  55 },
		{ 202,  54, 202,  54, 202,  54, 202,  54 }, { 203,  53, 203,  53, 203,  53, 203,  53 },
		{ 204,  52, 204,  52, 204,  52, 204,  52 }, { 205,  51, 205,  51, 205,  51, 205,  51 },
		{ 206,  50, 206,  50, 206,  50, 206,  50 }, { 207,  49, 207,  49, 207,  49, 207,  49 },
		{ 208,  48, 208,  48, 208,  48, 208,  48 }, { 209,  47, 209,  47, 209,  47, 209,  47 },
		{ 210,  46, 210,  46, 210,  46, 210,  46 }, { 211,  45, 211,  45, 211,  45, 211,  45 },
		{ 212,  44, 212,  44, 212,  44, 212,  44 }, { 213,  43, 213,  43, 213,  43, 213,  43 },
		{ 214,  42, 214,  42, 214,  42, 214,  42 }, { 215,  41, 215,  41, 215,  41, 215,  41 },
		{ 216,  40, 216,  40, 216,  40, 216,  40 }, { 217,  39, 217,  39, 217,  39, 217,  39 },
		{ 218,  38, 218,  38, 218,  38, 218,  38 }, { 219,  37, 219,  37, 219,  37, 219,  37 },
		{ 220,  36, 220,  36, 220,  36, 220,  36 }, { 221,  35, 221,  35, 221,  35, 221,  35 },
		{ 222,  34, 222,  34, 222,  34, 222,  34 }, { 223,  33, 223,  33, 223,  33, 223,  33 },
		{ 224,  32, 224,  32, 224,  32, 224,  32 }, { 225,  31, 225,  31, 225,  31, 225,  31 },
		{ 226,  30, 226,  30, 226,  30, 226,  30 }, { 227,  29, 227,  29, 227,  29, 227,  29 },
		{ 228,  28, 228,  28, 228,  28, 228,  28 }, { 229,  27, 229,  27, 229,  27, 229,  27 },
		{ 230,  26, 230,  26, 230,  26, 230,  26 }, { 231,  25, 231,  25, 231,  25, 231,  25 },
		{ 232,  24, 232,  24, 232,  24, 232,  24 }, { 233,  23, 233,  23, 233,  23, 233,  23 },
		{ 234,  22, 234,  22, 234,  22, 234,  22 }, { 235,  21, 235,  21, 235,  21, 235,  21 },
		{ 236,  20, 236,  20, 236,  20, 236,  20 }, { 237,  19, 237,  19, 237,  19, 237,  19 },
		{ 238,  18, 238,  18, 238,  18, 238,  18 }, { 239,  17, 239,  17, 239,  17, 239,  17 },
		{ 240,  16, 240,  16, 240,  16, 240,  16 }, { 241,  15, 241,  15, 241,  15, 241,  15 },
		{ 242,  14, 242,  14, 242,  14, 242,  14 }, { 243,  13, 243,  13, 243,  13, 243,  13 },
		{ 244,  12, 244,  12, 244,  12, 244,  12 }, { 245,  11, 245,  11, 245,  11, 245,  11 },
		{ 246,  10, 246,  10, 246,  10, 246,  10 }, { 247,   9, 247,   9, 247,   9, 247,   9 },
		{ 248,   8, 248,   8, 248,   8, 248,   8 }, { 249,   7, 249,   7, 249,   7, 249,   7 },
		{ 250,   6, 250,   6, 250,   6, 250,   6 }, { 251,   5, 251,   5, 251,   5, 251,   5 },
		{ 252,   4, 252,   4, 252,   4, 252,   4 }, { 253,   3, 253,   3, 253,   3, 253,   3 },
		{ 254,   2, 254,   2, 254,   2, 254,   2 }, { 255,   1, 255,   1, 255,   1, 255,   1 }
	}
};

__m128i scale_factor(u8 index)
{
	return *reinterpret_cast<const __m128i *>(&rgbutil_statics.scale_table[index][0]);
}

} // anonymous namespace

u32 rgbaint_t::bilinear_filter(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	__m128i color00 = _mm_cvtsi32_si128(rgb00);
	__m128i color01 = _mm_cvtsi32_si128(rgb01);
	__m128i color10 = _mm_cvtsi32_si128(rgb10);
	__m128i color11 = _mm_cvtsi32_si128(rgb11);

	// interleave color01 and color00 at the byte level
	color01 = _mm_unpacklo_epi8(color01, color00);
	color11 = _mm_unpacklo_epi8(color11, color10);
	color01 = _mm_unpacklo_epi8(color01, _mm_setzero_si128());
	color11 = _mm_unpacklo_epi8(color11, _mm_setzero_si128());
	color01 = _mm_madd_epi16(color01, scale_factor(u));
	color11 = _mm_madd_epi16(color11, scale_factor(u));
	color01 = _mm_slli_epi32(color01, 15);
	color11 = _mm_srli_epi32(color11, 1);
	color01 = _mm_max_epi16(color01, color11);
	color01 = _mm_madd_epi16(color01, scale_factor(v));
	color01 = _mm_srli_epi32(color01, 15);
	color01 = _mm_packs_epi32(color01, _mm_setzero_si128());
	color01 = _mm_packus_epi16(color01, _mm_setzero_si128());
	return _mm_cvtsi128_si32(color01);
}

void rgbaint_t::bilinear_filter_rgbaint(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	__m128i color00 = _mm_cvtsi32_si128(rgb00);
	__m128i color01 = _mm_cvtsi32_si128(rgb01);
	__m128i color10 = _mm_cvtsi32_si128(rgb10);
	__m128i color11 = _mm_cvtsi32_si128(rgb11);

	// interleave color01 and color00 at the byte level
	color01 = _mm_unpacklo_epi8(color01, color00);
	color11 = _mm_unpacklo_epi8(color11, color10);
	color01 = _mm_unpacklo_epi8(color01, _mm_setzero_si128());
	color11 = _mm_unpacklo_epi8(color11, _mm_setzero_si128());
	color01 = _mm_madd_epi16(color01, scale_factor(u));
	color11 = _mm_madd_epi16(color11, scale_factor(u));
	color01 = _mm_slli_epi32(color01, 15);
	color11 = _mm_srli_epi32(color11, 1);
	color01 = _mm_max_epi16(color01, color11);
	color01 = _mm_madd_epi16(color01, scale_factor(v));
	color01 = _mm_srli_epi32(color01, 15);

	m_a = _mm_extract_epi16(color01, 6);
	m_r = _mm_extract_epi16(color01, 4);
	m_g = _mm_extract_epi16(color01, 2);
	m_b = _mm_extract_epi16(color01, 0);
}

#elif defined(__ALTIVEC__)

#include <altivec.h>

namespace {

typedef __vector signed char    VECS8;
typedef __vector unsigned char  VECU8;
typedef __vector signed short   VECS16;
typedef __vector unsigned short VECU16;
typedef __vector signed int     VECS32;
typedef __vector unsigned int   VECU32;

const VECS16 scale_table[256] =
{
	{   0, 256,   0, 256,   0, 256,   0, 256 }, {   1, 255,   1, 255,   1, 255,   1, 255 },
	{   2, 254,   2, 254,   2, 254,   2, 254 }, {   3, 253,   3, 253,   3, 253,   3, 253 },
	{   4, 252,   4, 252,   4, 252,   4, 252 }, {   5, 251,   5, 251,   5, 251,   5, 251 },
	{   6, 250,   6, 250,   6, 250,   6, 250 }, {   7, 249,   7, 249,   7, 249,   7, 249 },
	{   8, 248,   8, 248,   8, 248,   8, 248 }, {   9, 247,   9, 247,   9, 247,   9, 247 },
	{  10, 246,  10, 246,  10, 246,  10, 246 }, {  11, 245,  11, 245,  11, 245,  11, 245 },
	{  12, 244,  12, 244,  12, 244,  12, 244 }, {  13, 243,  13, 243,  13, 243,  13, 243 },
	{  14, 242,  14, 242,  14, 242,  14, 242 }, {  15, 241,  15, 241,  15, 241,  15, 241 },
	{  16, 240,  16, 240,  16, 240,  16, 240 }, {  17, 239,  17, 239,  17, 239,  17, 239 },
	{  18, 238,  18, 238,  18, 238,  18, 238 }, {  19, 237,  19, 237,  19, 237,  19, 237 },
	{  20, 236,  20, 236,  20, 236,  20, 236 }, {  21, 235,  21, 235,  21, 235,  21, 235 },
	{  22, 234,  22, 234,  22, 234,  22, 234 }, {  23, 233,  23, 233,  23, 233,  23, 233 },
	{  24, 232,  24, 232,  24, 232,  24, 232 }, {  25, 231,  25, 231,  25, 231,  25, 231 },
	{  26, 230,  26, 230,  26, 230,  26, 230 }, {  27, 229,  27, 229,  27, 229,  27, 229 },
	{  28, 228,  28, 228,  28, 228,  28, 228 }, {  29, 227,  29, 227,  29, 227,  29, 227 },
	{  30, 226,  30, 226,  30, 226,  30, 226 }, {  31, 225,  31, 225,  31, 225,  31, 225 },
	{  32, 224,  32, 224,  32, 224,  32, 224 }, {  33, 223,  33, 223,  33, 223,  33, 223 },
	{  34, 222,  34, 222,  34, 222,  34, 222 }, {  35, 221,  35, 221,  35, 221,  35, 221 },
	{  36, 220,  36, 220,  36, 220,  36, 220 }, {  37, 219,  37, 219,  37, 219,  37, 219 },
	{  38, 218,  38, 218,  38, 218,  38, 218 }, {  39, 217,  39, 217,  39, 217,  39, 217 },
	{  40, 216,  40, 216,  40, 216,  40, 216 }, {  41, 215,  41, 215,  41, 215,  41, 215 },
	{  42, 214,  42, 214,  42, 214,  42, 214 }, {  43, 213,  43, 213,  43, 213,  43, 213 },
	{  44, 212,  44, 212,  44, 212,  44, 212 }, {  45, 211,  45, 211,  45, 211,  45, 211 },
	{  46, 210,  46, 210,  46, 210,  46, 210 }, {  47, 209,  47, 209,  47, 209,  47, 209 },
	{  48, 208,  48, 208,  48, 208,  48, 208 }, {  49, 207,  49, 207,  49, 207,  49, 207 },
	{  50, 206,  50, 206,  50, 206,  50, 206 }, {  51, 205,  51, 205,  51, 205,  51, 205 },
	{  52, 204,  52, 204,  52, 204,  52, 204 }, {  53, 203,  53, 203,  53, 203,  53, 203 },
	{  54, 202,  54, 202,  54, 202,  54, 202 }, {  55, 201,  55, 201,  55, 201,  55, 201 },
	{  56, 200,  56, 200,  56, 200,  56, 200 }, {  57, 199,  57, 199,  57, 199,  57, 199 },
	{  58, 198,  58, 198,  58, 198,  58, 198 }, {  59, 197,  59, 197,  59, 197,  59, 197 },
	{  60, 196,  60, 196,  60, 196,  60, 196 }, {  61, 195,  61, 195,  61, 195,  61, 195 },
	{  62, 194,  62, 194,  62, 194,  62, 194 }, {  63, 193,  63, 193,  63, 193,  63, 193 },
	{  64, 192,  64, 192,  64, 192,  64, 192 }, {  65, 191,  65, 191,  65, 191,  65, 191 },
	{  66, 190,  66, 190,  66, 190,  66, 190 }, {  67, 189,  67, 189,  67, 189,  67, 189 },
	{  68, 188,  68, 188,  68, 188,  68, 188 }, {  69, 187,  69, 187,  69, 187,  69, 187 },
	{  70, 186,  70, 186,  70, 186,  70, 186 }, {  71, 185,  71, 185,  71, 185,  71, 185 },
	{  72, 184,  72, 184,  72, 184,  72, 184 }, {  73, 183,  73, 183,  73, 183,  73, 183 },
	{  74, 182,  74, 182,  74, 182,  74, 182 }, {  75, 181,  75, 181,  75, 181,  75, 181 },
	{  76, 180,  76, 180,  76, 180,  76, 180 }, {  77, 179,  77, 179,  77, 179,  77, 179 },
	{  78, 178,  78, 178,  78, 178,  78, 178 }, {  79, 177,  79, 177,  79, 177,  79, 177 },
	{  80, 176,  80, 176,  80, 176,  80, 176 }, {  81, 175,  81, 175,  81, 175,  81, 175 },
	{  82, 174,  82, 174,  82, 174,  82, 174 }, {  83, 173,  83, 173,  83, 173,  83, 173 },
	{  84, 172,  84, 172,  84, 172,  84, 172 }, {  85, 171,  85, 171,  85, 171,  85, 171 },
	{  86, 170,  86, 170,  86, 170,  86, 170 }, {  87, 169,  87, 169,  87, 169,  87, 169 },
	{  88, 168,  88, 168,  88, 168,  88, 168 }, {  89, 167,  89, 167,  89, 167,  89, 167 },
	{  90, 166,  90, 166,  90, 166,  90, 166 }, {  91, 165,  91, 165,  91, 165,  91, 165 },
	{  92, 164,  92, 164,  92, 164,  92, 164 }, {  93, 163,  93, 163,  93, 163,  93, 163 },
	{  94, 162,  94, 162,  94, 162,  94, 162 }, {  95, 161,  95, 161,  95, 161,  95, 161 },
	{  96, 160,  96, 160,  96, 160,  96, 160 }, {  97, 159,  97, 159,  97, 159,  97, 159 },
	{  98, 158,  98, 158,  98, 158,  98, 158 }, {  99, 157,  99, 157,  99, 157,  99, 157 },
	{ 100, 156, 100, 156, 100, 156, 100, 156 }, { 101, 155, 101, 155, 101, 155, 101, 155 },
	{ 102, 154, 102, 154, 102, 154, 102, 154 }, { 103, 153, 103, 153, 103, 153, 103, 153 },
	{ 104, 152, 104, 152, 104, 152, 104, 152 }, { 105, 151, 105, 151, 105, 151, 105, 151 },
	{ 106, 150, 106, 150, 106, 150, 106, 150 }, { 107, 149, 107, 149, 107, 149, 107, 149 },
	{ 108, 148, 108, 148, 108, 148, 108, 148 }, { 109, 147, 109, 147, 109, 147, 109, 147 },
	{ 110, 146, 110, 146, 110, 146, 110, 146 }, { 111, 145, 111, 145, 111, 145, 111, 145 },
	{ 112, 144, 112, 144, 112, 144, 112, 144 }, { 113, 143, 113, 143, 113, 143, 113, 143 },
	{ 114, 142, 114, 142, 114, 142, 114, 142 }, { 115, 141, 115, 141, 115, 141, 115, 141 },
	{ 116, 140, 116, 140, 116, 140, 116, 140 }, { 117, 139, 117, 139, 117, 139, 117, 139 },
	{ 118, 138, 118, 138, 118, 138, 118, 138 }, { 119, 137, 119, 137, 119, 137, 119, 137 },
	{ 120, 136, 120, 136, 120, 136, 120, 136 }, { 121, 135, 121, 135, 121, 135, 121, 135 },
	{ 122, 134, 122, 134, 122, 134, 122, 134 }, { 123, 133, 123, 133, 123, 133, 123, 133 },
	{ 124, 132, 124, 132, 124, 132, 124, 132 }, { 125, 131, 125, 131, 125, 131, 125, 131 },
	{ 126, 130, 126, 130, 126, 130, 126, 130 }, { 127, 129, 127, 129, 127, 129, 127, 129 },
	{ 128, 128, 128, 128, 128, 128, 128, 128 }, { 129, 127, 129, 127, 129, 127, 129, 127 },
	{ 130, 126, 130, 126, 130, 126, 130, 126 }, { 131, 125, 131, 125, 131, 125, 131, 125 },
	{ 132, 124, 132, 124, 132, 124, 132, 124 }, { 133, 123, 133, 123, 133, 123, 133, 123 },
	{ 134, 122, 134, 122, 134, 122, 134, 122 }, { 135, 121, 135, 121, 135, 121, 135, 121 },
	{ 136, 120, 136, 120, 136, 120, 136, 120 }, { 137, 119, 137, 119, 137, 119, 137, 119 },
	{ 138, 118, 138, 118, 138, 118, 138, 118 }, { 139, 117, 139, 117, 139, 117, 139, 117 },
	{ 140, 116, 140, 116, 140, 116, 140, 116 }, { 141, 115, 141, 115, 141, 115, 141, 115 },
	{ 142, 114, 142, 114, 142, 114, 142, 114 }, { 143, 113, 143, 113, 143, 113, 143, 113 },
	{ 144, 112, 144, 112, 144, 112, 144, 112 }, { 145, 111, 145, 111, 145, 111, 145, 111 },
	{ 146, 110, 146, 110, 146, 110, 146, 110 }, { 147, 109, 147, 109, 147, 109, 147, 109 },
	{ 148, 108, 148, 108, 148, 108, 148, 108 }, { 149, 107, 149, 107, 149, 107, 149, 107 },
	{ 150, 106, 150, 106, 150, 106, 150, 106 }, { 151, 105, 151, 105, 151, 105, 151, 105 },
	{ 152, 104, 152, 104, 152, 104, 152, 104 }, { 153, 103, 153, 103, 153, 103, 153, 103 },
	{ 154, 102, 154, 102, 154, 102, 154, 102 }, { 155, 101, 155, 101, 155, 101, 155, 101 },
	{ 156, 100, 156, 100, 156, 100, 156, 100 }, { 157,  99, 157,  99, 157,  99, 157,  99 },
	{ 158,  98, 158,  98, 158,  98, 158,  98 }, { 159,  97, 159,  97, 159,  97, 159,  97 },
	{ 160,  96, 160,  96, 160,  96, 160,  96 }, { 161,  95, 161,  95, 161,  95, 161,  95 },
	{ 162,  94, 162,  94, 162,  94, 162,  94 }, { 163,  93, 163,  93, 163,  93, 163,  93 },
	{ 164,  92, 164,  92, 164,  92, 164,  92 }, { 165,  91, 165,  91, 165,  91, 165,  91 },
	{ 166,  90, 166,  90, 166,  90, 166,  90 }, { 167,  89, 167,  89, 167,  89, 167,  89 },
	{ 168,  88, 168,  88, 168,  88, 168,  88 }, { 169,  87, 169,  87, 169,  87, 169,  87 },
	{ 170,  86, 170,  86, 170,  86, 170,  86 }, { 171,  85, 171,  85, 171,  85, 171,  85 },
	{ 172,  84, 172,  84, 172,  84, 172,  84 }, { 173,  83, 173,  83, 173,  83, 173,  83 },
	{ 174,  82, 174,  82, 174,  82, 174,  82 }, { 175,  81, 175,  81, 175,  81, 175,  81 },
	{ 176,  80, 176,  80, 176,  80, 176,  80 }, { 177,  79, 177,  79, 177,  79, 177,  79 },
	{ 178,  78, 178,  78, 178,  78, 178,  78 }, { 179,  77, 179,  77, 179,  77, 179,  77 },
	{ 180,  76, 180,  76, 180,  76, 180,  76 }, { 181,  75, 181,  75, 181,  75, 181,  75 },
	{ 182,  74, 182,  74, 182,  74, 182,  74 }, { 183,  73, 183,  73, 183,  73, 183,  73 },
	{ 184,  72, 184,  72, 184,  72, 184,  72 }, { 185,  71, 185,  71, 185,  71, 185,  71 },
	{ 186,  70, 186,  70, 186,  70, 186,  70 }, { 187,  69, 187,  69, 187,  69, 187,  69 },
	{ 188,  68, 188,  68, 188,  68, 188,  68 }, { 189,  67, 189,  67, 189,  67, 189,  67 },
	{ 190,  66, 190,  66, 190,  66, 190,  66 }, { 191,  65, 191,  65, 191,  65, 191,  65 },
	{ 192,  64, 192,  64, 192,  64, 192,  64 }, { 193,  63, 193,  63, 193,  63, 193,  63 },
	{ 194,  62, 194,  62, 194,  62, 194,  62 }, { 195,  61, 195,  61, 195,  61, 195,  61 },
	{ 196,  60, 196,  60, 196,  60, 196,  60 }, { 197,  59, 197,  59, 197,  59, 197,  59 },
	{ 198,  58, 198,  58, 198,  58, 198,  58 }, { 199,  57, 199,  57, 199,  57, 199,  57 },
	{ 200,  56, 200,  56, 200,  56, 200,  56 }, { 201,  55, 201,  55, 201,  55, 201,  55 },
	{ 202,  54, 202,  54, 202,  54, 202,  54 }, { 203,  53, 203,  53, 203,  53, 203,  53 },
	{ 204,  52, 204,  52, 204,  52, 204,  52 }, { 205,  51, 205,  51, 205,  51, 205,  51 },
	{ 206,  50, 206,  50, 206,  50, 206,  50 }, { 207,  49, 207,  49, 207,  49, 207,  49 },
	{ 208,  48, 208,  48, 208,  48, 208,  48 }, { 209,  47, 209,  47, 209,  47, 209,  47 },
	{ 210,  46, 210,  46, 210,  46, 210,  46 }, { 211,  45, 211,  45, 211,  45, 211,  45 },
	{ 212,  44, 212,  44, 212,  44, 212,  44 }, { 213,  43, 213,  43, 213,  43, 213,  43 },
	{ 214,  42, 214,  42, 214,  42, 214,  42 }, { 215,  41, 215,  41, 215,  41, 215,  41 },
	{ 216,  40, 216,  40, 216,  40, 216,  40 }, { 217,  39, 217,  39, 217,  39, 217,  39 },
	{ 218,  38, 218,  38, 218,  38, 218,  38 }, { 219,  37, 219,  37, 219,  37, 219,  37 },
	{ 220,  36, 220,  36, 220,  36, 220,  36 }, { 221,  35, 221,  35, 221,  35, 221,  35 },
	{ 222,  34, 222,  34, 222,  34, 222,  34 }, { 223,  33, 223,  33, 223,  33, 223,  33 },
	{ 224,  32, 224,  32, 224,  32, 224,  32 }, { 225,  31, 225,  31, 225,  31, 225,  31 },
	{ 226,  30, 226,  30, 226,  30, 226,  30 }, { 227,  29, 227,  29, 227,  29, 227,  29 },
	{ 228,  28, 228,  28, 228,  28, 228,  28 }, { 229,  27, 229,  27, 229,  27, 229,  27 },
	{ 230,  26, 230,  26, 230,  26, 230,  26 }, { 231,  25, 231,  25, 231,  25, 231,  25 },
	{ 232,  24, 232,  24, 232,  24, 232,  24 }, { 233,  23, 233,  23, 233,  23, 233,  23 },
	{ 234,  22, 234,  22, 234,  22, 234,  22 }, { 235,  21, 235,  21, 235,  21, 235,  21 },
	{ 236,  20, 236,  20, 236,  20, 236,  20 }, { 237,  19, 237,  19, 237,  19, 237,  19 },
	{ 238,  18, 238,  18, 238,  18, 238,  18 }, { 239,  17, 239,  17, 239,  17, 239,  17 },
	{ 240,  16, 240,  16, 240,  16, 240,  16 }, { 241,  15, 241,  15, 241,  15, 241,  15 },
	{ 242,  14, 242,  14, 242,  14, 242,  14 }, { 243,  13, 243,  13, 243,  13, 243,  13 },
	{ 244,  12, 244,  12, 244,  12, 244,  12 }, { 245,  11, 245,  11, 245,  11, 245,  11 },
	{ 246,  10, 246,  10, 246,  10, 246,  10 }, { 247,   9, 247,   9, 247,   9, 247,   9 },
	{ 248,   8, 248,   8, 248,   8, 248,   8 }, { 249,   7, 249,   7, 249,   7, 249,   7 },
	{ 250,   6, 250,   6, 250,   6, 250,   6 }, { 251,   5, 251,   5, 251,   5, 251,   5 },
	{ 252,   4, 252,   4, 252,   4, 252,   4 }, { 253,   3, 253,   3, 253,   3, 253,   3 },
	{ 254,   2, 254,   2, 254,   2, 254,   2 }, { 255,   1, 255,   1, 255,   1, 255,   1 }
};

} // anonymous namespace

u32 rgbaint_t::bilinear_filter(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	const VECS32 zero = vec_splat_s32(0);

	// put each packed value into first element of a vector register
#ifdef __LITTLE_ENDIAN__
	VECS32 color00 = *reinterpret_cast<const VECS32 *>(&rgb00);
	VECS32 color01 = *reinterpret_cast<const VECS32 *>(&rgb01);
	VECS32 color10 = *reinterpret_cast<const VECS32 *>(&rgb10);
	VECS32 color11 = *reinterpret_cast<const VECS32 *>(&rgb11);
#else
	VECS32 color00 = vec_perm(VECS32(vec_lde(0, &rgb00)), zero, vec_lvsl(0, &rgb00));
	VECS32 color01 = vec_perm(VECS32(vec_lde(0, &rgb01)), zero, vec_lvsl(0, &rgb01));
	VECS32 color10 = vec_perm(VECS32(vec_lde(0, &rgb10)), zero, vec_lvsl(0, &rgb10));
	VECS32 color11 = vec_perm(VECS32(vec_lde(0, &rgb11)), zero, vec_lvsl(0, &rgb11));
#endif

	// interleave color01/color00 and color10/color11 at the byte level then zero-extend
	color01 = VECS32(vec_mergeh(VECU8(color01), VECU8(color00)));
	color11 = VECS32(vec_mergeh(VECU8(color11), VECU8(color10)));
#ifdef __LITTLE_ENDIAN__
	color01 = VECS32(vec_mergeh(VECU8(color01), VECU8(zero)));
	color11 = VECS32(vec_mergeh(VECU8(color11), VECU8(zero)));
#else
	color01 = VECS32(vec_mergeh(VECU8(zero), VECU8(color01)));
	color11 = VECS32(vec_mergeh(VECU8(zero), VECU8(color11)));
#endif

	color01 = vec_msum(VECS16(color01), scale_table[u], zero);
	color11 = vec_msum(VECS16(color11), scale_table[u], zero);
	color01 = vec_sl(color01, vec_splat_u32(15));
	color11 = vec_sr(color11, vec_splat_u32(1));
	color01 = VECS32(vec_max(VECS16(color01), VECS16(color11)));
	color01 = vec_msum(VECS16(color01), scale_table[v], zero);
	color01 = vec_sr(color01, vec_splat_u32(15));
	color01 = VECS32(vec_packs(color01, color01));
	color01 = VECS32(vec_packsu(VECS16(color01), VECS16(color01)));

	u32 result;
	vec_ste(VECU32(color01), 0, &result);
	return result;
}

void rgbaint_t::bilinear_filter_rgbaint(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	const VECS32 zero = vec_splat_s32(0);

	// put each packed value into first element of a vector register
#ifdef __LITTLE_ENDIAN__
	VECS32 color00 = *reinterpret_cast<const VECS32 *>(&rgb00);
	VECS32 color01 = *reinterpret_cast<const VECS32 *>(&rgb01);
	VECS32 color10 = *reinterpret_cast<const VECS32 *>(&rgb10);
	VECS32 color11 = *reinterpret_cast<const VECS32 *>(&rgb11);
#else
	VECS32 color00 = vec_perm(VECS32(vec_lde(0, &rgb00)), zero, vec_lvsl(0, &rgb00));
	VECS32 color01 = vec_perm(VECS32(vec_lde(0, &rgb01)), zero, vec_lvsl(0, &rgb01));
	VECS32 color10 = vec_perm(VECS32(vec_lde(0, &rgb10)), zero, vec_lvsl(0, &rgb10));
	VECS32 color11 = vec_perm(VECS32(vec_lde(0, &rgb11)), zero, vec_lvsl(0, &rgb11));
#endif

	// interleave color01/color00 and color10/color11 at the byte level then zero-extend
	color01 = VECS32(vec_mergeh(VECU8(color01), VECU8(color00)));
	color11 = VECS32(vec_mergeh(VECU8(color11), VECU8(color10)));
#ifdef __LITTLE_ENDIAN__
	color01 = VECS32(vec_mergeh(VECU8(color01), VECU8(zero)));
	color11 = VECS32(vec_mergeh(VECU8(color11), VECU8(zero)));
#else
	color01 = VECS32(vec_mergeh(VECU8(zero), VECU8(color01)));
	color11 = VECS32(vec_mergeh(VECU8(zero), VECU8(color11)));
#endif

	color01 = vec_msum(VECS16(color01), scale_table[u], zero);
	color11 = vec_msum(VECS16(color11), scale_table[u], zero);
	color01 = vec_sl(color01, vec_splat_u32(15));
	color11 = vec_sr(color11, vec_splat_u32(1));
	color01 = VECS32(vec_max(VECS16(color01), VECS16(color11)));
	color01 = vec_msum(VECS16(color01), scale_table[v], zero);
	color01 = vec_sr(color01, vec_splat_u32(15));

#ifdef __LITTLE_ENDIAN__
	vec_ste(vec_splat(color01, 3), 0, &m_a);
	vec_ste(vec_splat(color01, 2), 0, &m_r);
	vec_ste(vec_splat(color01, 1), 0, &m_g);
	vec_ste(vec_splat(color01, 0), 0, &m_b);
#else
	vec_ste(vec_splat(color01, 0), 0, &m_a);
	vec_ste(vec_splat(color01, 1), 0, &m_r);
	vec_ste(vec_splat(color01, 2), 0, &m_g);
	vec_ste(vec_splat(color01, 3), 0, &m_b);
#endif
}

#elif defined(__ARM_NEON)

#include <arm_neon.h>

u32 rgbaint_t::bilinear_filter(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	// interpolate on u axis
	const uint16x8_t colorx0 = vmulq_n_u16(vmovl_u8(vcreate_u8((u64(rgb10) << 32) | rgb00)), 256U - u);
	const uint16x8_t colorx1 = vmulq_n_u16(vmovl_u8(vcreate_u8((u64(rgb11) << 32) | rgb01)), u);
	const uint32x4_t color0x = vaddq_u32(vmovl_u16(vget_low_u16(colorx0)), vmovl_u16(vget_low_u16(colorx1)));
	const uint32x4_t color1x = vaddq_u32(vmovl_u16(vget_high_u16(colorx0)), vmovl_u16(vget_high_u16(colorx1)));

	// interpolate on v axis
	const uint32x4_t color = vaddq_u32(vmulq_n_u32(color0x, 256U - v), vmulq_n_u32(color1x, v));

	// scale and saturate
	const uint16x4_t color16 = vqmovn_u32(vshrq_n_u32(color, 16));
	return vget_lane_u32(vreinterpret_u32_u8(vqmovn_u16(vcombine_u16(color16, color16))), 0);
}

void rgbaint_t::bilinear_filter_rgbaint(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	// interpolate on u axis
	const uint16x8_t colorx0 = vmulq_n_u16(vmovl_u8(vcreate_u8((u64(rgb10) << 32) | rgb00)), 256U - u);
	const uint16x8_t colorx1 = vmulq_n_u16(vmovl_u8(vcreate_u8((u64(rgb11) << 32) | rgb01)), u);
	const uint32x4_t color0x = vaddq_u32(vmovl_u16(vget_low_u16(colorx0)), vmovl_u16(vget_low_u16(colorx1)));
	const uint32x4_t color1x = vaddq_u32(vmovl_u16(vget_high_u16(colorx0)), vmovl_u16(vget_high_u16(colorx1)));

	// interpolate on v axis and scale
	const uint32x4_t color = vshrq_n_u32(vaddq_u32(vmulq_n_u32(color0x, 256U - v), vmulq_n_u32(color1x, v)), 16);

	m_a = s32(vgetq_lane_u32(color, 3));
	m_r = s32(vgetq_lane_u32(color, 2));
	m_g = s32(vgetq_lane_u32(color, 1));
	m_b = s32(vgetq_lane_u32(color, 0));
}

#else

u32 rgbaint_t::bilinear_filter(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	// interpolate on u axis
	const u32 ag00 = ((rgb00 >> 8) & 0x00ff00ff) * (256U  - u);
	const u32 rb00 = ((rgb00 >> 0) & 0x00ff00ff) * (256U  - u);
	const u32 ag10 = ((rgb10 >> 8) & 0x00ff00ff) * (256U  - u);
	const u32 rb10 = ((rgb10 >> 0) & 0x00ff00ff) * (256U  - u);
	const u32 ag01 = ((rgb01 >> 8) & 0x00ff00ff) * u;
	const u32 rb01 = ((rgb01 >> 0) & 0x00ff00ff) * u;
	const u32 ag11 = ((rgb11 >> 8) & 0x00ff00ff) * u;
	const u32 rb11 = ((rgb11 >> 0) & 0x00ff00ff) * u;
	const u32 ag0x = ((ag00 >> 1) & 0x7fff7fff) + ((ag01 >> 1) & 0x7fff7fff);
	const u32 rb0x = ((rb00 >> 1) & 0x7fff7fff) + ((rb01 >> 1) & 0x7fff7fff);
	const u32 ag1x = ((ag10 >> 1) & 0x7fff7fff) + ((ag11 >> 1) & 0x7fff7fff);
	const u32 rb1x = ((rb10 >> 1) & 0x7fff7fff) + ((rb11 >> 1) & 0x7fff7fff);

	// interpolate on v axis
	const u32 a = (((ag0x >> 16) * (256U - v)) + ((ag1x >> 16) * v)) >> 15;
	const u32 r = (((rb0x >> 16) * (256U - v)) + ((rb1x >> 16) * v)) >> 15;
	const u32 g = (((ag0x & 0xffffU) * (256U - v)) + ((ag1x & 0xffffU) * v)) >> 15;
	const u32 b = (((rb0x & 0xffffU) * (256U - v)) + ((rb1x & 0xffffU) * v)) >> 15;

	return (a << 24) | (r << 16) | (g << 8) | (b << 0);
}

void rgbaint_t::bilinear_filter_rgbaint(u32 rgb00, u32 rgb01, u32 rgb10, u32 rgb11, u8 u, u8 v) noexcept
{
	// interpolate on u axis
	const u32 ag00 = ((rgb00 >> 8) & 0x00ff00ff) * (256U  - u);
	const u32 rb00 = ((rgb00 >> 0) & 0x00ff00ff) * (256U  - u);
	const u32 ag10 = ((rgb10 >> 8) & 0x00ff00ff) * (256U  - u);
	const u32 rb10 = ((rgb10 >> 0) & 0x00ff00ff) * (256U  - u);
	const u32 ag01 = ((rgb01 >> 8) & 0x00ff00ff) * u;
	const u32 rb01 = ((rgb01 >> 0) & 0x00ff00ff) * u;
	const u32 ag11 = ((rgb11 >> 8) & 0x00ff00ff) * u;
	const u32 rb11 = ((rgb11 >> 0) & 0x00ff00ff) * u;
	const u32 ag0x = ((ag00 >> 1) & 0x7fff7fff) + ((ag01 >> 1) & 0x7fff7fff);
	const u32 rb0x = ((rb00 >> 1) & 0x7fff7fff) + ((rb01 >> 1) & 0x7fff7fff);
	const u32 ag1x = ((ag10 >> 1) & 0x7fff7fff) + ((ag11 >> 1) & 0x7fff7fff);
	const u32 rb1x = ((rb10 >> 1) & 0x7fff7fff) + ((rb11 >> 1) & 0x7fff7fff);

	// interpolate on v axis
	m_a = (((ag0x >> 16) * (256U - v)) + ((ag1x >> 16) * v)) >> 15;
	m_r = (((rb0x >> 16) * (256U - v)) + ((rb1x >> 16) * v)) >> 15;
	m_g = (((ag0x & 0xffffU) * (256U - v)) + ((ag1x & 0xffffU) * v)) >> 15;
	m_b = (((rb0x & 0xffffU) * (256U - v)) + ((rb1x & 0xffffU) * v)) >> 15;
}

#endif
