﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rum/CloudWatchRUMRequest.h>
#include <aws/rum/CloudWatchRUM_EXPORTS.h>
#include <aws/rum/model/MetricDestination.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace CloudWatchRUM {
namespace Model {

/**
 */
class BatchGetRumMetricDefinitionsRequest : public CloudWatchRUMRequest {
 public:
  AWS_CLOUDWATCHRUM_API BatchGetRumMetricDefinitionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchGetRumMetricDefinitions"; }

  AWS_CLOUDWATCHRUM_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHRUM_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the CloudWatch RUM app monitor that is sending the metrics.</p>
   */
  inline const Aws::String& GetAppMonitorName() const { return m_appMonitorName; }
  inline bool AppMonitorNameHasBeenSet() const { return m_appMonitorNameHasBeenSet; }
  template <typename AppMonitorNameT = Aws::String>
  void SetAppMonitorName(AppMonitorNameT&& value) {
    m_appMonitorNameHasBeenSet = true;
    m_appMonitorName = std::forward<AppMonitorNameT>(value);
  }
  template <typename AppMonitorNameT = Aws::String>
  BatchGetRumMetricDefinitionsRequest& WithAppMonitorName(AppMonitorNameT&& value) {
    SetAppMonitorName(std::forward<AppMonitorNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of destination that you want to view metrics for. Valid values are
   * <code>CloudWatch</code> and <code>Evidently</code>.</p>
   */
  inline MetricDestination GetDestination() const { return m_destination; }
  inline bool DestinationHasBeenSet() const { return m_destinationHasBeenSet; }
  inline void SetDestination(MetricDestination value) {
    m_destinationHasBeenSet = true;
    m_destination = value;
  }
  inline BatchGetRumMetricDefinitionsRequest& WithDestination(MetricDestination value) {
    SetDestination(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is required if <code>Destination</code> is
   * <code>Evidently</code>. If <code>Destination</code> is <code>CloudWatch</code>,
   * do not use this parameter.</p> <p>This parameter specifies the ARN of the
   * Evidently experiment that corresponds to the destination.</p>
   */
  inline const Aws::String& GetDestinationArn() const { return m_destinationArn; }
  inline bool DestinationArnHasBeenSet() const { return m_destinationArnHasBeenSet; }
  template <typename DestinationArnT = Aws::String>
  void SetDestinationArn(DestinationArnT&& value) {
    m_destinationArnHasBeenSet = true;
    m_destinationArn = std::forward<DestinationArnT>(value);
  }
  template <typename DestinationArnT = Aws::String>
  BatchGetRumMetricDefinitionsRequest& WithDestinationArn(DestinationArnT&& value) {
    SetDestinationArn(std::forward<DestinationArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in one operation. The default is 50.
   * The maximum that you can specify is 100.</p> <p>To retrieve the remaining
   * results, make another call with the returned <code>NextToken</code> value. </p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline BatchGetRumMetricDefinitionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Use the token returned by the previous operation to request the next page of
   * results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  BatchGetRumMetricDefinitionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appMonitorName;

  MetricDestination m_destination{MetricDestination::NOT_SET};

  Aws::String m_destinationArn;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_appMonitorNameHasBeenSet = false;
  bool m_destinationHasBeenSet = false;
  bool m_destinationArnHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchRUM
}  // namespace Aws
