"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.promqlSearchStrategyProvider = void 0;
var _datemath = _interopRequireDefault(require("@elastic/datemath"));
var _common = require("../../../data/common");
var _prometheus_manager = require("../connections/managers/prometheus_manager");
var _prom_utils = require("./prom_utils");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// MAX_SERIES_TABLE: Maximum series for table display
const MAX_SERIES_TABLE = 2000;
// MAX_SERIES_VIZ: Maximum series for visualization. This should be lower than MAX_SERIES_TABLE
const MAX_SERIES_VIZ = 100;
// We'll want to re-evaluate this when we provide an affordance for step configuration
const MAX_DATAPOINTS = _prom_utils.DEFAULT_RESOLUTION * MAX_SERIES_TABLE;

/**
 * Result from executing a single query in a multi-query context
 */

const promqlSearchStrategyProvider = (config$, logger, usage) => {
  return {
    search: async (context, request, options) => {
      try {
        var _requestBody$step, _dataset$id, _queryResults$;
        const {
          body: requestBody
        } = request;
        const parsedFrom = _datemath.default.parse(requestBody.timeRange.from);
        const parsedTo = _datemath.default.parse(requestBody.timeRange.to, {
          roundUp: true
        });
        if (!parsedFrom || !parsedTo) {
          throw new Error('Invalid time range format');
        }
        const timeRange = {
          start: parsedFrom.unix(),
          end: parsedTo.unix()
        };
        const durationMs = (timeRange.end - timeRange.start) * 1000;
        const step = (_requestBody$step = requestBody.step) !== null && _requestBody$step !== void 0 ? _requestBody$step : (0, _prom_utils.calculateStep)(durationMs);
        const {
          dataset,
          query,
          language
        } = requestBody.query;
        const datasetId = (_dataset$id = dataset === null || dataset === void 0 ? void 0 : dataset.id) !== null && _dataset$id !== void 0 ? _dataset$id : '';
        const parsedQueries = (0, _common.splitMultiQueries)(query);
        const isSingleQuery = parsedQueries.length === 1;

        // Execute all queries uniformly
        const queryResults = await executeMultipleQueries(context, request, parsedQueries, {
          language,
          maxResults: Math.floor(MAX_DATAPOINTS / parsedQueries.length),
          timeout: 30,
          timeRange,
          step: step.toString()
        }, datasetId, logger);

        // For single query, preserve fail-fast behavior
        if (isSingleQuery && (_queryResults$ = queryResults[0]) !== null && _queryResults$ !== void 0 && _queryResults$.error) {
          throw new Error(queryResults[0].error);
        }
        const dataFrame = createDataFrame(queryResults, datasetId, isSingleQuery);
        return {
          type: _common.DATA_FRAME_TYPES.DEFAULT,
          body: dataFrame
        };
      } catch (e) {
        const error = e;
        logger.error(`promqlSearchStrategy: ${error.message}`);
        if (usage) usage.trackError();
        throw e;
      }
    }
  };
};

/**
 * Executes multiple PromQL queries in parallel
 */
exports.promqlSearchStrategyProvider = promqlSearchStrategyProvider;
async function executeMultipleQueries(context, request, queries, options, datasetId, logger) {
  const promises = queries.map(async parsedQuery => {
    const params = {
      body: {
        query: parsedQuery.query,
        language: options.language,
        maxResults: options.maxResults,
        timeout: options.timeout,
        options: {
          queryType: 'range',
          start: options.timeRange.start.toString(),
          end: options.timeRange.end.toString(),
          step: options.step
        }
      },
      dataconnection: datasetId
    };
    try {
      const queryRes = await _prometheus_manager.prometheusManager.query(context, request, params);
      return {
        label: parsedQuery.label,
        response: queryRes
      };
    } catch (error) {
      var _body;
      let errorMessage = `Query ${parsedQuery.label} failed`;
      if (error instanceof Error) {
        errorMessage = error.message;
      }

      // Try to extract detailed error from response body from SQL plugin
      const responseBody = (_body = error === null || error === void 0 ? void 0 : error.body) !== null && _body !== void 0 ? _body : error === null || error === void 0 ? void 0 : error.response;
      if (responseBody) {
        try {
          var _ref, _ref2, _parsed$error$details, _parsed$error, _parsed$error2, _parsed$error3;
          const parsed = typeof responseBody === 'string' ? JSON.parse(responseBody) : responseBody;
          errorMessage = (_ref = (_ref2 = (_parsed$error$details = parsed === null || parsed === void 0 || (_parsed$error = parsed.error) === null || _parsed$error === void 0 ? void 0 : _parsed$error.details) !== null && _parsed$error$details !== void 0 ? _parsed$error$details : parsed === null || parsed === void 0 || (_parsed$error2 = parsed.error) === null || _parsed$error2 === void 0 ? void 0 : _parsed$error2.reason) !== null && _ref2 !== void 0 ? _ref2 : parsed === null || parsed === void 0 || (_parsed$error3 = parsed.error) === null || _parsed$error3 === void 0 ? void 0 : _parsed$error3.message) !== null && _ref !== void 0 ? _ref : errorMessage;
        } catch {
          // error might come from other places, use original message if failed
        }
      }
      return {
        label: parsedQuery.label,
        error: errorMessage
      };
    }
  });
  return Promise.all(promises);
}

/**
 * Formats metric labels into a string like {label1="value1", label2="value2"}
 * Only includes labels that have actual values (non-empty)
 */
function formatMetricLabels(metric) {
  const labelParts = Object.entries(metric).filter(([_, value]) => value !== undefined && value !== '').sort(([a], [b]) => a.localeCompare(b)).map(([key, value]) => `${key}="${value}"`);
  return `{${labelParts.join(', ')}}`;
}

/**
 * Unified DataFrame creation for single and multi-query results.
 * - Single query: uses `Value` column, simpler series names
 * - Multi-query: uses `Value #A`, `Value #B` columns, prefixed series names
 */
function createDataFrame(queryResults, datasetId, isSingleQuery) {
  const allVizRows = [];
  const allLabelKeys = new Set();

  // instantDataMap is used for table display, we only show the latest datapoint in the table.
  const instantDataMap = new Map();
  const queryLabels = queryResults.filter(r => !r.error && r.response).map(r => r.label);
  queryResults.forEach(result => {
    var _result$response$resu;
    if (!result.response || result.error) return;
    const series = ((_result$response$resu = result.response.results[datasetId]) === null || _result$response$resu === void 0 ? void 0 : _result$response$resu.result) || [];
    series.forEach((metricResult, i) => {
      if (i >= MAX_SERIES_TABLE) return;
      if (metricResult.metric) {
        Object.keys(metricResult.metric).forEach(key => {
          if (key !== '__name__') {
            allLabelKeys.add(key);
          }
        });
      }
    });
  });
  const labelKeys = Array.from(allLabelKeys).sort();
  queryResults.forEach(result => {
    var _result$response$resu2;
    if (!result.response || result.error) return;
    const series = ((_result$response$resu2 = result.response.results[datasetId]) === null || _result$response$resu2 === void 0 ? void 0 : _result$response$resu2.result) || [];
    series.forEach((metricResult, seriesIndex) => {
      if (seriesIndex >= MAX_SERIES_TABLE) return;
      const metricName = metricResult.metric.__name__ || '';
      const labelsWithoutName = {
        ...metricResult.metric
      };
      delete labelsWithoutName.__name__;
      const formattedLabels = formatMetricLabels(metricResult.metric);
      const seriesName = isSingleQuery ? formattedLabels : `${result.label}: ${formattedLabels}`;
      // TODO: remove escaping if not using vega
      // Escape brackets in series name to prevent Vega's splitAccessPath from
      // interpreting them as array index notation when used as field names
      const escapedSeriesName = seriesName.replace(/\[/g, '\\[').replace(/\]/g, '\\]');
      metricResult.values.forEach(([timestamp, value]) => {
        const metricSignature = JSON.stringify({
          name: metricName,
          labels: labelsWithoutName
        });
        const existing = instantDataMap.get(metricSignature);
        const timeMs = timestamp * 1000;
        if (!existing || timeMs > existing.time) {
          instantDataMap.set(metricSignature, {
            metric: labelsWithoutName,
            metricName,
            time: timeMs,
            valuesByQuery: {
              ...((existing === null || existing === void 0 ? void 0 : existing.valuesByQuery) || {}),
              [result.label]: Number(value)
            }
          });
        } else if (timeMs === existing.time) {
          existing.valuesByQuery[result.label] = Number(value);
        }
        if (seriesIndex < MAX_SERIES_VIZ) {
          allVizRows.push({
            Time: timeMs,
            Series: escapedSeriesName,
            Value: Number(value)
          });
        }
      });
    });
  });
  const allInstantRows = [];
  instantDataMap.forEach(data => {
    const row = {
      Time: data.time,
      Metric: data.metricName + formatMetricLabels(data.metric)
    };
    labelKeys.forEach(labelKey => {
      const labelValue = data.metric[labelKey];
      row[labelKey] = labelValue !== undefined && labelValue !== '' ? labelValue : undefined;
    });
    if (isSingleQuery) {
      var _data$valuesByQuery$q;
      row.Value = (_data$valuesByQuery$q = data.valuesByQuery[queryLabels[0]]) !== null && _data$valuesByQuery$q !== void 0 ? _data$valuesByQuery$q : null;
    } else {
      queryLabels.forEach(label => {
        var _data$valuesByQuery$l;
        row[`Value #${label}`] = (_data$valuesByQuery$l = data.valuesByQuery[label]) !== null && _data$valuesByQuery$l !== void 0 ? _data$valuesByQuery$l : null;
      });
    }
    allInstantRows.push(row);
  });
  const valueColumns = isSingleQuery ? [{
    name: 'Value',
    type: 'number',
    values: []
  }] : queryLabels.map(label => ({
    name: `Value #${label}`,
    type: 'number',
    values: []
  }));
  const instantSchema = [{
    name: 'Time',
    type: 'time',
    values: []
  }, {
    name: 'Metric',
    type: 'string',
    values: []
  }, ...labelKeys.map(key => ({
    name: key,
    type: 'string',
    values: []
  })), ...valueColumns];
  const vizSchema = [{
    name: 'Time',
    type: 'time',
    values: []
  }, {
    name: 'Series',
    type: 'string',
    values: []
  }, {
    name: 'Value',
    type: 'number',
    values: []
  }];
  const vizFields = vizSchema.map(s => ({
    name: s.name,
    type: s.type,
    values: allVizRows.map(row => row[s.name])
  }));
  const errors = queryResults.filter(r => r.error).map(r => ({
    query: r.label,
    error: r.error
  }));
  const meta = {
    instantData: {
      schema: instantSchema,
      rows: allInstantRows
    }
  };
  if (!isSingleQuery) {
    meta.multiQuery = {
      queryCount: queryResults.length,
      successCount: queryResults.filter(r => !r.error).length,
      errors,
      queryLabels
    };
  }
  return {
    type: _common.DATA_FRAME_TYPES.DEFAULT,
    name: datasetId,
    schema: vizSchema,
    fields: vizFields,
    size: allVizRows.length,
    meta
  };
}