/*
 * Copyright (C) 2003-2014 The Music Player Daemon Project
 * http://www.musicpd.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MPD_DECODER_LIST_HXX
#define MPD_DECODER_LIST_HXX

#include "Compiler.h"

struct DecoderPlugin;

extern const struct DecoderPlugin *const decoder_plugins[];
extern bool decoder_plugins_enabled[];

/* interface for using plugins */

gcc_pure
const struct DecoderPlugin *
decoder_plugin_from_name(const char *name);

/* this is where we "load" all the "plugins" ;-) */
void decoder_plugin_init_all(void);

/* this is where we "unload" all the "plugins" */
void decoder_plugin_deinit_all(void);

template<typename F>
static inline const DecoderPlugin *
decoder_plugins_find(F f)
{
	for (unsigned i = 0; decoder_plugins[i] != nullptr; ++i)
		if (decoder_plugins_enabled[i] && f(*decoder_plugins[i]))
			return decoder_plugins[i];

	return nullptr;
}

template<typename F>
static inline bool
decoder_plugins_try(F f)
{
	for (unsigned i = 0; decoder_plugins[i] != nullptr; ++i)
		if (decoder_plugins_enabled[i] && f(*decoder_plugins[i]))
			return true;

	return false;
}

template<typename F>
static inline void
decoder_plugins_for_each(F f)
{
	for (auto i = decoder_plugins; *i != nullptr; ++i)
		f(**i);
}

template<typename F>
static inline void
decoder_plugins_for_each_enabled(F f)
{
	for (unsigned i = 0; decoder_plugins[i] != nullptr; ++i)
		if (decoder_plugins_enabled[i])
			f(*decoder_plugins[i]);
}

/**
 * Is there at least once #DecoderPlugin that supports the specified
 * file name suffix?
 */
gcc_pure gcc_nonnull_all
bool
decoder_plugins_supports_suffix(const char *suffix);

#endif
