#region License, Terms and Conditions
//
// Jayrock - JSON and JSON-RPC for Microsoft .NET Framework and Mono
// Written by Atif Aziz (www.raboof.com)
// Copyright (c) 2005 Atif Aziz. All rights reserved.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License as published by the Free
// Software Foundation; either version 3 of the License, or (at your option)
// any later version.
//
// This library is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
//
#endregion

namespace Jayrock.JsonRpc.Web
{
    #region Imports

    using System;
    using System.Diagnostics;
    using System.Globalization;
    using System.IO;
    using Jayrock.Json.Conversion;
    using Jayrock.Services;

    #endregion

    public sealed class JsonRpcProxyGenerator : JsonRpcProxyGeneratorBase
    {
        public JsonRpcProxyGenerator(IService service) : 
            base(service) {}

        protected override string ContentType
        {
            get { return "text/javascript"; }
        }

        protected override string ClientFileName
        {
            get { return Service.GetClass().Name + "Proxy.js"; }
        }
        
        protected override void WriteProxy(IndentedTextWriter writer)
        {
            if (writer == null)
                throw new ArgumentNullException("writer");

            writer.WriteLine("// This JavaScript was automatically generated by");
            writer.Write("// ");
            writer.WriteLine(GetType().AssemblyQualifiedName);
            writer.Write("// on ");
            DateTime now = DateTime.Now;
            TimeZone timeZone = TimeZone.CurrentTimeZone;
            writer.Write(now.ToLongDateString());
            writer.Write(" at ");
            writer.Write(now.ToLongTimeString());
            writer.Write(" (");
            writer.Write(timeZone.IsDaylightSavingTime(now) ? 
                timeZone.DaylightName : timeZone.StandardName);
            writer.WriteLine(")");
            writer.WriteLine();

            string version = Mask.EmptyString(Request.QueryString["v"], "1").Trim();

            Uri url = new Uri(Request.Url.GetLeftPart(UriPartial.Path));
            
            if (version.Equals("2"))
                Version2(Service.GetClass(), url, writer);
            else
                Version1(Service.GetClass(), url, writer);
        }

        public static void Generate(ServiceClass service, Uri url, TextWriter writer)
        {
            if (service == null) throw new ArgumentNullException("service");
            if (url == null) throw new ArgumentNullException("url");
            if (writer == null) throw new ArgumentNullException("writer");

            Version1(service, url, new IndentedTextWriter(writer));
        }

        private static void Version1(ServiceClass service, Uri url, IndentedTextWriter writer)
        {
            Debug.Assert(service != null);
            Debug.Assert(url!= null);
            Debug.Assert(!url.IsFile);
            Debug.Assert(writer != null);

            writer.Write("function ");
            writer.Write(service.Name);
            writer.WriteLine("(url)");
            writer.WriteLine("{");
            writer.Indent++;
            writer.WriteLine("var self = this;");
            
            Method[] methods = service.GetMethods();
            
            string[] methodNames = new string[methods.Length];
            bool[] idempotents = new bool[methods.Length];

            for (int i = 0; i < methods.Length; i++)
            {
                methodNames[i] = methods[i].Name;
                idempotents[i] = methods[i].Idempotent;
            }

            writer.Write("var m = ");
            JsonConvert.Export(methodNames, writer);
            writer.WriteLine(';');
            writer.Write("var idems = ");
            JsonConvert.Export(idempotents, writer);
            writer.WriteLine(';');
            writer.WriteLine();
    
            for (int i = 0; i < methods.Length; i++)
            {
                Method method = methods[i];
                string index = i.ToString(CultureInfo.InvariantCulture);

                if (method.Description.Length > 0)
                {
                    // TODO: What to do if summary breaks over several lines?

                    writer.Write("// ");
                    writer.WriteLine(method.Description);
                    writer.WriteLine();
                }

                writer.Write("this[m[");
                writer.Write(index);
                writer.Write("]] = function /* "); 
                writer.Write(method.Name);
                writer.Write(" */ (");

                Parameter[] parameters = method.GetParameters();
                
                foreach (Parameter parameter in parameters)
                {
                    writer.Write(parameter.Name);
                    writer.Write(", ");
                }

                writer.WriteLine("callback)");
                writer.WriteLine("{");
                writer.Indent++;

                writer.Write("if (self.kwargs) return rpc(new Call(");
                writer.Write(index);
                writer.Write(", {");

                foreach (Parameter parameter in parameters)
                {
                    if (parameter.Position > 0)
                        writer.Write(',');
                    writer.Write(' ');
                    writer.Write(parameter.Name);
                    writer.Write(": ");
                    writer.Write(parameter.Name);
                }

                writer.WriteLine(" }, callback));");

                writer.Write("return rpc(new Call(");
                writer.Write(index);
                writer.Write(", [");

                foreach (Parameter parameter in parameters)
                {
                    if (parameter.Position > 0)
                        writer.Write(',');
                    writer.Write(' ');
                    writer.Write(parameter.Name);
                }

                writer.WriteLine(" ], callback));");

                writer.Indent--;
                writer.WriteLine("}");
                writer.WriteLine();
            }
    
            writer.Write("var url = typeof(url) === 'string' ? url : '");
            writer.Write(url);
            writer.WriteLine("';");
            writer.WriteLine(@"var nextId = 0;

    function Call(method, params, callback)
    {
        this.url = url;
        this.callback = callback;
        this.proxy = self;
        this.idempotent = idems[method];
        this.request = 
        { 
            id     : ++nextId, 
            method : m[method], 
            params : params 
        };
    }

    function rpc(call)
    {
        return self.channel != null && typeof(self.channel.rpc) === 'function' ?
            self.channel.rpc(call) : call;
    }

    this.kwargs = false;
    this.channel = new JayrockChannel();

    function JayrockChannel()
    {
        this.rpc = function(call)
        {
            var async = typeof(call.callback) === 'function';
            var xhr = newXHR();
            xhr.open('POST', call.url, async, this.httpUserName, this.httpPassword);
            xhr.setRequestHeader('Content-Type', this.contentType || 'application/json; charset=utf-8');
            xhr.setRequestHeader('X-JSON-RPC', call.request.method);
            if (async) xhr.onreadystatechange = function() { xhr_onreadystatechange(xhr, call.callback); }
            xhr.send(JSON.stringify(call.request));
            call.handler = xhr;
            if (async) return call;
            if (xhr.status != 200) throw new Error(xhr.status + ' ' + xhr.statusText);
            var response = JSON.parse(xhr.responseText);
            if (response.error != null) throw response.error;
            return response.result;
        }

        function xhr_onreadystatechange(sender, callback)
        {
            if (sender.readyState == /* complete */ 4)
            {
                try { 
                    sender.onreadystatechange = null; // Avoid IE7 leak (bug #12964)
                } 
                catch (e) { 
                    /* IE 6/Mobile throws for onreadystatechange = null */ 
                }

                var response = sender.status == 200 ? 
                    JSON.parse(sender.responseText) : {};
                
                callback(response, sender);
            }
        }

        function newXHR()
        {
            if (typeof(window) !== 'undefined' && window.XMLHttpRequest)
                return new XMLHttpRequest(); /* IE7, Safari 1.2, Mozilla 1.0/Firefox, and Netscape 7 */
            else
                return new ActiveXObject('Microsoft.XMLHTTP'); /* WSH and IE 5 to IE 6 */
        }
    }");
    
            writer.Indent--;
            writer.WriteLine("}");
    
            writer.WriteLine();
            writer.Write(service.Name);
            writer.Write(".rpcMethods = ");
            JsonConvert.Export(methodNames, writer);
            writer.WriteLine(";");
        }

        private void Version2(ServiceClass service, Uri url, IndentedTextWriter writer)
        {
            Debug.Assert(service != null);
            Debug.Assert(url!= null);
            Debug.Assert(!url.IsFile);
            Debug.Assert(writer != null);
            
            if (JsonRpcTrace.TraceWarning)
            {
                writer.WriteLine(@"// *** O B S O L E T E *** 
//
// You are using version 2 of the JavaScript proxy that MAY 
// be obsoleted and removed in the next release. The functionality has 
// been integrated into the original proxy version via the channel 
// property on the JavaScript proxy.
");
            }
 
            writer.WriteLine("// Proxy version 2.0");
            writer.WriteLine();
 
            writer.Write("var ");
            writer.Write(service.Name);
            writer.Write(@" = function()
{
    var nextId = 0;

    var proxy = {

        url : """);
            writer.Write(url);
            writer.Write(@""",
        rpc : {");
            writer.WriteLine();
            writer.Indent += 3;
    
            Method[] methods = service.GetMethods();
            
            string[] methodNames = new string[methods.Length];
            for (int i = 0; i < methods.Length; i++)
                methodNames[i] = methods[i].Name;
            
            Array.Sort(methodNames, methods);
    
            for (int i = 0; i < methods.Length; i++)
            {
                Method method = methods[i];

                writer.WriteLine();

                if (method.Description.Length > 0)
                {
                    // TODO: What to do if /* and */ appear in the summary?

                    writer.Write("/* ");
                    writer.Write(method.Description);
                    writer.WriteLine(" */");
                    writer.WriteLine();
                }

                writer.Write('\"');
                writer.Write(method.Name);
                writer.Write("\" : function(");

                Parameter[] parameters = method.GetParameters();
                
                foreach (Parameter parameter in parameters)
                {
                    writer.Write(parameter.Name);
                    writer.Write(", ");
                }

                writer.WriteLine("callback) {");
                writer.Indent++;

                writer.Write("return new Call(\"");
                writer.Write(method.Name);
                writer.Write("\", [");

                foreach (Parameter parameter in parameters)
                {
                    if (parameter.Position > 0)
                        writer.Write(',');
                    writer.Write(' ');
                    writer.Write(parameter.Name);
                }

                writer.WriteLine(" ], callback);");

                writer.Indent--;
                writer.Write("}");
                if (i < (methods.Length - 1))
                    writer.Write(',');
                writer.WriteLine();
            }
    
            writer.Indent--;
            writer.WriteLine(@"}
    }

    function Call(method, params, callback)
    {
        this.url = proxy.url;
        this.callback = callback;
        this.request = 
        { 
            id     : ++nextId, 
            method : method, 
            params : params
        };
    }
    
    Call.prototype.call = function(channel) { return channel(this); }
    
    return proxy;
}();");
    
            writer.Indent--;
        }
    }
}
