import subprocess
from PySide6.QtGui import QIcon
from PySide6.QtWidgets import (
    QProgressDialog, QPushButton, QVBoxLayout, QDialog, QLabel, QComboBox, QHBoxLayout, QMessageBox
)
from PySide6.QtCore import Qt
import time
from collections import Counter

from src.controller.switch_validate_controller import SwitchValidateController
from src.path_resolver import PathResolver
from src.controller.terminal_controller import TerminalController


class DockerController:
    """
    Provides static methods to manage Docker containers and images.

    This class employs a caching strategy to minimize expensive calls to the Docker daemon.
    The status of Docker, installed images, and running containers is _fetched once_ and
    stored in a class-level cache (`_docker_cache`), which is only refreshed after a
    short TTL interval has passed.
    """
    _docker_check_cache = {
        "last_refresh": 0.0,
        "docker_running": False,
        "installed_images": set(),
        "installed_repos": set(),
        "running_images": Counter(),
        "running_repos": Counter()
    }
    # Defines the minimum time interval (in seconds) before the cache is considered stale.
    _cache_min_interval = 2.0

    def __init__(self, context):
        self.tools_list = context.ui.tools_list
        self.tools = context.tools
        self.tool_cards = context.ui.tool_cards
        self.switch_validator = SwitchValidateController()
        self.docker_status_label = context.ui.docker_status_label
        self.docker_status_icon = context.ui.docker_status_icon

    @staticmethod
    def _now() -> float:
        return time.monotonic()

    @staticmethod
    def _normalize_repo_name(image_ref: str) -> str:
        """
        Normalize a Docker image reference to a repository path without registry and tag.

        Examples:
        - docker.io/parrotsec/nmap:latest -> parrotsec/nmap
        - parrotsec/nmap:latest -> parrotsec/nmap
        """
        ref = image_ref.strip()
        if '@' in ref:
            ref = ref.split('@', 1)[0]
        last_slash = ref.rfind('/')
        last_colon = ref.rfind(':')
        if last_colon > last_slash:
            ref_no_tag = ref[:last_colon]
        else:
            ref_no_tag = ref
        # Remove the registry domain if present (heuristic check for '.' or ':')
        parts = ref_no_tag.split('/')
        if parts and ('.' in parts[0] or ':' in parts[0]):
            parts = parts[1:]
        return '/'.join(parts) if parts else ref_no_tag

    @staticmethod
    def refresh_cache(force: bool = False) -> None:
        """
        This method aims to answer the following questions:

        - Is Docker running?
        - What are the downloaded Docker images?
        - What are the running Docker containers?
        """
        now = DockerController._now()
        if (not force and
                (now - DockerController._docker_check_cache["last_refresh"]) < DockerController._cache_min_interval):
            return

        docker_running = False
        installed_images = set()
        installed_repos = set()
        running_images = Counter()
        running_repos = Counter()

        try:
            subprocess.run(['docker', 'info'], stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True)
            docker_running = True

            if docker_running:
                try:
                    result = subprocess.run(
                        ['docker', 'images', '--format', '{{.Repository}}'],
                        stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True
                    )
                    lines = result.stdout.decode('utf-8').splitlines()
                    installed_images = set(line.strip() for line in lines if line.strip())
                    installed_repos = set(DockerController._normalize_repo_name(repo) for repo in installed_images)
                except subprocess.CalledProcessError:
                    pass

                try:
                    result = subprocess.run(
                        ['docker', 'ps', '--format', '{{.Image}}'],
                        stdout=subprocess.PIPE, stderr=subprocess.PIPE, check=True
                    )
                    imgs = [line.strip() for line in result.stdout.decode('utf-8').splitlines() if line.strip()]
                    running_images = Counter(imgs)
                    running_repos = Counter(DockerController._normalize_repo_name(img) for img in imgs)
                except subprocess.CalledProcessError:
                    pass

        except (subprocess.CalledProcessError, FileNotFoundError):
            pass

        DockerController._docker_check_cache.update({
            "last_refresh": now,
            "docker_running": docker_running,
            "installed_images": installed_images,
            "installed_repos": installed_repos,
            "running_images": running_images,
            "running_repos": running_repos,
        })

    @staticmethod
    def run_container(context):
        """
        Run the Docker container for the selected tool with specified flags.
        """
        current_item = context.ui.tools_list.currentItem()
        if current_item is not None and current_item.parent() is not None:
            parent = current_item.parent()
            current_index = parent.indexOfChild(current_item)
            current_tool = context.tools[current_index]
            current_card = context.ui.tool_cards[current_index]

            docker_image = current_tool["docker_image"]
            default_port = current_tool.get("default_port")
            flags = current_card.get_params()
            tool_name = current_tool["name"].lower()

            switch_validator = SwitchValidateController()
            if not switch_validator.validate_switch(flags, tool_name):
                msg = QMessageBox()
                msg.setIcon(QMessageBox.Icon.Critical)
                msg.setText("Invalid flags")
                msg.setWindowTitle("Error")
                msg.exec()
                return

            command = f'docker run --rm -it {default_port if default_port else ""} {docker_image} {flags}'
            TerminalController.run(command)
        else:
            msg = QMessageBox()
            msg.setIcon(QMessageBox.Icon.Warning)
            msg.setText("Please select a valid tool.")
            msg.setInformativeText(
                "You need to select a tool from the list to run the Docker container.")
            msg.setWindowTitle("Warning")
            msg.setStandardButtons(QMessageBox.StandardButton.Ok)
            msg.exec()

    @staticmethod
    def check_container_status(tool, tool_item, run_button):
        """
        Check if Docker is running and if the Docker image is downloaded,
        then update the status and run button labels.
        """
        docker_image = tool["docker_image"]

        DockerController.refresh_cache()
        cache = DockerController._docker_check_cache
        if not cache["docker_running"]:
            return

        target_repo = DockerController._normalize_repo_name(docker_image)

        installed = target_repo in cache["installed_repos"]
        try:
            if installed:
                tool_item.setIcon(0, QIcon(PathResolver.resource_path('resources/assets/check.png')))
            else:
                tool_item.setIcon(0, QIcon(PathResolver.resource_path('resources/assets/cross.png')))
        except Exception:
            pass

        try:
            running_count = cache["running_repos"].get(target_repo, 0)
            if running_count > 1:
                run_button.setText("Containers Running")
            elif running_count == 1:
                run_button.setText("Container Running")
            else:
                run_button.setText("Run Container")
        except Exception as e:
            run_button.setText(f"Container status check is not possible\n {e}")

    @staticmethod
    def stop_container(tool, parent_widget):
        """
        Stop the Docker container associated with the specified tool by displaying a progress dialog.
        """
        docker_image = tool["docker_image"]
        progress_dialog = QProgressDialog("Stopping container...", "Cancel", 0, 100, parent_widget)
        progress_dialog.setWindowModality(Qt.WindowModality.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setAutoReset(False)
        progress_dialog.show()

        try:
            result = subprocess.run(['docker', 'ps', '-q', '-f', f'ancestor={docker_image}'],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            container_ids = result.stdout.strip().decode('utf-8').split('\n')
            progress_dialog.setValue(30)

            if container_ids and container_ids[0]:
                container_id = None
                if len(container_ids) > 1:
                    dialog = QDialog(parent_widget)
                    dialog.setWindowTitle("Select Container")

                    layout = QVBoxLayout()

                    label = QLabel("Choose container ID to stop:")
                    layout.addWidget(label)

                    combo_box = QComboBox()
                    combo_box.addItems([f"{cid}" for cid in container_ids])
                    layout.addWidget(combo_box)

                    button_layout = QHBoxLayout()

                    stop_all_button = QPushButton("Stop All Containers")
                    ok_button = QPushButton("Ok")
                    cancel_button = QPushButton("Cancel")

                    button_layout.addWidget(stop_all_button)
                    button_layout.addWidget(cancel_button)
                    button_layout.addWidget(ok_button)

                    layout.addLayout(button_layout)

                    def handle_stop_all():
                        dialog.accept()
                        DockerController.stop_all_containers(tool, parent_widget)

                    stop_all_button.clicked.connect(handle_stop_all)
                    ok_button.clicked.connect(dialog.accept)
                    cancel_button.clicked.connect(dialog.reject)

                    dialog.setLayout(layout)
                    dialog.exec()

                    if dialog.result() == QDialog.DialogCode.Accepted:
                        item = combo_box.currentText()
                        parts = item.split(': ')
                        if len(parts) > 0:
                            container_id = parts[0]
                        else:
                            print(f"Unexpected format for selected item: {item}")
                else:
                    container_id = container_ids[0]

                if container_id:
                    stop_process = subprocess.run(['docker', 'stop', container_id],
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.PIPE,
                                                  check=True)
                    progress_dialog.setValue(50)
                    if stop_process.returncode == 0:
                        print(f"Container {container_id} stopped successfully.")
                    else:
                        print(f"Failed to stop container {container_id}.")
                    progress_dialog.setValue(100)
                else:
                    print("No container selected to stop.")
            else:
                print("No running container found for this tool.")
        except subprocess.CalledProcessError as e:
            print(f"Failed to stop container: {e}")
        finally:
            progress_dialog.close()

    @staticmethod
    def stop_all_containers(tool, parent_widget):
        """
        Stop all the Docker containers associated with the specified tool by displaying a progress dialog.
        """
        docker_image = tool["docker_image"]
        progress_dialog = QProgressDialog("Stopping all containers...", "Cancel", 0, 100, parent_widget)
        progress_dialog.setWindowModality(Qt.WindowModality.WindowModal)
        progress_dialog.setAutoClose(False)
        progress_dialog.setAutoReset(False)
        progress_dialog.show()

        try:
            result = subprocess.run(['docker', 'ps', '-q', '-f', f'ancestor={docker_image}'],
                                    stdout=subprocess.PIPE,
                                    stderr=subprocess.PIPE,
                                    check=True)
            container_ids = result.stdout.strip().decode('utf-8').split('\n')
            progress_dialog.setValue(30)

            if container_ids and container_ids[0]:
                for i, container_id in enumerate(container_ids):
                    stop_process = subprocess.run(['docker', 'stop', container_id],
                                                  stdout=subprocess.PIPE,
                                                  stderr=subprocess.PIPE,
                                                  check=True)
                    progress_dialog.setValue(30 + int(70 * (i + 1) / len(container_ids)))
                    if stop_process.returncode == 0:
                        print(f"Container {container_id} stopped successfully.")
                    else:
                        print(f"Failed to stop container {container_id}.")
                progress_dialog.setValue(100)
            else:
                print("No running container found for this tool.")
        except subprocess.CalledProcessError as e:
            print(f"Failed to stop containers: {e}")
        finally:
            progress_dialog.close()
